<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\AI\ConversationService;
use App\Services\AI\Phi3Service;
use App\Models\User;

class CreateTestConversationCommand extends Command
{
    protected $signature = 'ai:create-test-conversation';
    protected $description = 'Create a test conversation to demonstrate AI features';

    public function handle()
    {
        $this->info('🎭 Creating Test Conversation...');
        $this->line('==============================');

        try {
            // Get the first user
            $user = User::first();
            if (!$user) {
                $this->error('No users found. Please create a user first.');
                return;
            }

            $this->line("Creating conversation for user: {$user->name} (ID: {$user->id})");

            // Create conversation service
            $conversationService = new ConversationService(app(Phi3Service::class));

            // Start a sales conversation
            $conversation = $conversationService->startConversation(
                $user->id,
                null, // no contact
                'sales',
                ['test' => true, 'demo' => 'AI Voice Agent CRM']
            );

            $this->info("✅ Conversation created with ID: {$conversation->id}");

            // Simulate a conversation
            $messages = [
                "Hello! I'm interested in your AI voice agent service. Can you tell me more about it?",
                "That sounds interesting. What are the main benefits for a CRM system?",
                "What about pricing? Do you have different packages?",
                "I'm concerned about data security. How do you handle that?",
                "That's reassuring. What's the implementation timeline?",
                "Perfect! I'd like to schedule a demo. When would be a good time?"
            ];

            $this->line('');
            $this->info('💬 Simulating conversation...');

            foreach ($messages as $index => $message) {
                $this->line("User: {$message}");
                
                $result = $conversationService->processMessage(
                    $conversation->id,
                    $message,
                    ['message_number' => $index + 1]
                );

                if ($result['success']) {
                    $aiResponse = $result['ai_message']['content'];
                    $this->line("AI: {$aiResponse}");
                    
                    // Show insights if any
                    if (!empty($result['insights'])) {
                        $this->line("Insights: " . json_encode($result['insights'], JSON_PRETTY_PRINT));
                    }
                } else {
                    $this->error("AI Response failed: " . $result['error']);
                }
                
                $this->line('');
            }

            // Complete the conversation
            $conversationService->completeConversation(
                $conversation->id,
                'demo_scheduled',
                'Customer interested in AI voice agent service, demo scheduled'
            );

            $this->info('✅ Test conversation completed!');
            $this->line("Conversation ID: {$conversation->id}");
            $this->line("You can now view this conversation in the AI dashboard at /user/ai");

        } catch (\Exception $e) {
            $this->error('❌ Error: ' . $e->getMessage());
            $this->line('Stack trace: ' . $e->getTraceAsString());
        }
    }
}
