<?php
// Simple API with Complete Authentication
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');
header('Content-Type: application/json');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Database configuration
$host = 'localhost';
$username = 'root';
$password = '';
$database = 'dizaartdemo_api';

try {
    $pdo = new PDO("mysql:host=$host;dbname=$database;charset=utf8", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch(PDOException $e) {
    echo json_encode(['status' => 'error', 'message' => 'Database connection failed: ' . $e->getMessage()]);
    exit();
}

// JWT Secret Key (Change this in production)
define('JWT_SECRET', 'DizaartDemo_API_2025!@#SecureKey$%^&*StrongPassword123');
define('JWT_ALGORITHM', 'HS256');

// Get the request method and URI
$method = $_SERVER['REQUEST_METHOD'];
$request = $_SERVER['REQUEST_URI'];
$path = parse_url($request, PHP_URL_PATH);

// Remove the base path and filename
$basePath = '/ci3_api/simple_api.php';
if (strpos($path, $basePath) === 0) {
    $path = substr($path, strlen($basePath));
}

// Remove leading slash
$path = ltrim($path, '/');

// JWT Functions
function generate_jwt($payload) {
    $header = json_encode(['typ' => 'JWT', 'alg' => JWT_ALGORITHM]);
    $payload = json_encode($payload);
    
    $base64UrlHeader = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($header));
    $base64UrlPayload = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($payload));
    
    $signature = hash_hmac('sha256', $base64UrlHeader . "." . $base64UrlPayload, JWT_SECRET, true);
    $base64UrlSignature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));
    
    return $base64UrlHeader . "." . $base64UrlPayload . "." . $base64UrlSignature;
}

function verify_jwt($jwt) {
    try {
        $tokenParts = explode('.', $jwt);
        if (count($tokenParts) != 3) {
            return false;
        }
        
        $header = base64_decode(str_replace(['-', '_'], ['+', '/'], $tokenParts[0]));
        $payload = base64_decode(str_replace(['-', '_'], ['+', '/'], $tokenParts[1]));
        $signatureProvided = $tokenParts[2];
        
        $base64UrlHeader = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($header));
        $base64UrlPayload = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($payload));
        $signature = hash_hmac('sha256', $base64UrlHeader . "." . $base64UrlPayload, JWT_SECRET, true);
        $base64UrlSignature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));
        
        if ($base64UrlSignature !== $signatureProvided) {
            return false;
        }
        
        $payload = json_decode($payload, true);
        
        // Check if token is expired
        if (isset($payload['exp']) && $payload['exp'] < time()) {
            return false;
        }
        
        return $payload;
    } catch (Exception $e) {
        return false;
    }
}

function authenticate() {
    $headers = getallheaders();
    $auth_header = isset($headers['Authorization']) ? $headers['Authorization'] : (isset($headers['authorization']) ? $headers['authorization'] : '');
    
    if (empty($auth_header)) {
        http_response_code(401);
        echo json_encode(['status' => 'error', 'message' => 'Authorization header required']);
        return false;
    }
    
    $token = str_replace('Bearer ', '', $auth_header);
    $decoded = verify_jwt($token);
    
    if (!$decoded) {
        http_response_code(401);
        echo json_encode(['status' => 'error', 'message' => 'Invalid or expired token']);
        return false;
    }
    
    return $decoded;
}

function has_role($user, $allowed_roles) {
    return in_array($user['role_id'], $allowed_roles);
}

function validate_input($required_fields, $input) {
    $errors = [];
    foreach ($required_fields as $field) {
        if (!isset($input[$field]) || empty(trim($input[$field]))) {
            $errors[] = "$field is required";
        }
    }
    return $errors;
}

// Route handling
switch ($method) {
    case 'GET':
        if ($path === '' || $path === '/') {
            // API info
            echo json_encode([
                'status' => 'success',
                'message' => 'Secure Simple API is running',
                'endpoints' => [
                    'GET /' => 'API information',
                    'GET /users' => 'Get all users (Admin only)',
                    'GET /users/{id}' => 'Get user by ID',
                    'GET /products' => 'Get all products',
                    'GET /products/{id}' => 'Get product by ID',
                    'GET /service-categories' => 'Get service categories',
                    'GET /categories/{id}/subcategories' => 'Get subcategories',
                    'GET /cart' => 'Get user cart (Auth required)',
                    'GET /orders' => 'Get user orders (Auth required)',
                    'GET /wallet' => 'Get wallet balance (Auth required)',
                    'POST /register' => 'Register new user',
                    'POST /login' => 'User login',
                    'POST /verify-otp' => 'Verify OTP',
                    'POST /products' => 'Create product (Vendor only)',
                    'POST /cart' => 'Add to cart (Auth required)',
                    'POST /orders' => 'Create order (Auth required)'
                ]
            ]);
        } elseif ($path === 'users') {
            // Get all users (Admin only)
            $user = authenticate();
            if (!$user) return;
            
            if (!has_role($user, [3])) { // Only admin
                http_response_code(403);
                echo json_encode(['status' => 'error', 'message' => 'Access denied. Admin only.']);
                return;
            }
            
            $stmt = $pdo->query("
                SELECT u.id, u.phone, u.email, u.role_id, u.is_verified, u.status, u.created_at, 
                       up.full_name, r.role_name 
                FROM users u 
                LEFT JOIN user_profiles up ON u.id = up.user_id 
                LEFT JOIN roles r ON u.role_id = r.id
                ORDER BY u.created_at DESC
            ");
            $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
            echo json_encode(['status' => 'success', 'data' => $users]);
            
        } elseif (preg_match('/^users\/(\d+)$/', $path, $matches)) {
            // Get user by ID
            $user = authenticate();
            if (!$user) return;
            
            $id = $matches[1];
            
            // Users can only access their own data unless they're admin
            if ($user['user_id'] != $id && !has_role($user, [3])) {
                http_response_code(403);
                echo json_encode(['status' => 'error', 'message' => 'Access denied']);
                return;
            }
            
            $stmt = $pdo->prepare("
                SELECT u.id, u.phone, u.email, u.role_id, u.is_verified, u.status, u.created_at, 
                       up.full_name, up.father_name, up.dob, up.house_number, up.street, up.city, up.state, 
                       up.postal_code, up.profile_image, r.role_name 
                FROM users u 
                LEFT JOIN user_profiles up ON u.id = up.user_id 
                LEFT JOIN roles r ON u.role_id = r.id 
                WHERE u.id = ?
            ");
            $stmt->execute([$id]);
            $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($user_data) {
                unset($user_data['password_hash']);
                echo json_encode(['status' => 'success', 'data' => $user_data]);
            } else {
                http_response_code(404);
                echo json_encode(['status' => 'error', 'message' => 'User not found']);
            }
            
        } elseif ($path === 'products') {
            // Get all products (Public)
            $category_id = isset($_GET['category_id']) ? intval($_GET['category_id']) : null;
            $vendor_id = isset($_GET['vendor_id']) ? intval($_GET['vendor_id']) : null;
            $search = isset($_GET['search']) ? $_GET['search'] : null;
            $limit = isset($_GET['limit']) ? intval($_GET['limit']) : 20;
            $offset = isset($_GET['offset']) ? intval($_GET['offset']) : 0;
            
            $sql = "
                SELECT p.*, sc.name as category_name, sub.name as sub_category_name, 
                       u.phone as vendor_phone, up.full_name as vendor_name 
                FROM products p 
                LEFT JOIN service_categories sc ON p.category_id = sc.id 
                LEFT JOIN sub_categories sub ON p.sub_category_id = sub.id 
                LEFT JOIN users u ON p.vendor_user_id = u.id 
                LEFT JOIN user_profiles up ON u.id = up.user_id 
                WHERE p.is_active = 1
            ";
            
            $params = [];
            
            if ($category_id) {
                $sql .= " AND p.category_id = ?";
                $params[] = $category_id;
            }
            
            if ($vendor_id) {
                $sql .= " AND p.vendor_user_id = ?";
                $params[] = $vendor_id;
            }
            
            if ($search) {
                $sql .= " AND (p.title LIKE ? OR p.short_description LIKE ? OR p.description LIKE ?)";
                $search_term = "%$search%";
                $params[] = $search_term;
                $params[] = $search_term;
                $params[] = $search_term;
            }
            
            $sql .= " ORDER BY p.created_at DESC LIMIT ? OFFSET ?";
            $params[] = $limit;
            $params[] = $offset;
            
            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);
            $products = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            echo json_encode([
                'status' => 'success', 
                'data' => $products,
                'pagination' => [
                    'limit' => $limit,
                    'offset' => $offset,
                    'total' => count($products)
                ]
            ]);
            
        } elseif (preg_match('/^products\/(\d+)$/', $path, $matches)) {
            // Get product by ID (Public)
            $id = $matches[1];
            
            $stmt = $pdo->prepare("
                SELECT p.*, sc.name as category_name, sub.name as sub_category_name, 
                       u.phone as vendor_phone, up.full_name as vendor_name,
                       s.shop_name, s.shop_address, s.city as shop_city
                FROM products p 
                LEFT JOIN service_categories sc ON p.category_id = sc.id 
                LEFT JOIN sub_categories sub ON p.sub_category_id = sub.id 
                LEFT JOIN users u ON p.vendor_user_id = u.id 
                LEFT JOIN user_profiles up ON u.id = up.user_id 
                LEFT JOIN shops s ON u.id = s.user_id 
                WHERE p.id = ? AND p.is_active = 1
            ");
            $stmt->execute([$id]);
            $product = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($product) {
                echo json_encode(['status' => 'success', 'data' => $product]);
            } else {
                http_response_code(404);
                echo json_encode(['status' => 'error', 'message' => 'Product not found']);
            }
            
        } elseif ($path === 'service-categories') {
            // Get service categories (Public)
            $stmt = $pdo->query("SELECT * FROM service_categories WHERE is_active = 1 ORDER BY name");
            $categories = $stmt->fetchAll(PDO::FETCH_ASSOC);
            echo json_encode(['status' => 'success', 'data' => $categories]);
            
        } elseif (preg_match('/^categories\/(\d+)\/subcategories$/', $path, $matches)) {
            // Get subcategories by category ID (Public)
            $category_id = $matches[1];
            
            $stmt = $pdo->prepare("
                SELECT * FROM sub_categories 
                WHERE category_id = ? AND is_active = 1 
                ORDER BY name
            ");
            $stmt->execute([$category_id]);
            $subcategories = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            echo json_encode(['status' => 'success', 'data' => $subcategories]);
            
        } elseif ($path === 'cart') {
            // Get user cart (Auth required)
            $user = authenticate();
            if (!$user) return;
            
            $stmt = $pdo->prepare("
                SELECT c.*, p.title, p.image_url, p.price as unit_price,
                       (c.quantity * c.price) as total_price
                FROM cart c 
                LEFT JOIN products p ON c.product_id = p.id 
                WHERE c.user_id = ?
            ");
            $stmt->execute([$user['user_id']]);
            $cart_items = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $total = 0;
            foreach ($cart_items as $item) {
                $total += $item['total_price'];
            }
            
            echo json_encode([
                'status' => 'success', 
                'data' => $cart_items,
                'summary' => [
                    'total_items' => count($cart_items),
                    'total_amount' => $total
                ]
            ]);
            
        } elseif ($path === 'orders') {
            // Get user orders (Auth required)
            $user = authenticate();
            if (!$user) return;
            
            $stmt = $pdo->prepare("
                SELECT o.*, COUNT(oi.id) as item_count 
                FROM orders o 
                LEFT JOIN order_items oi ON o.id = oi.order_id 
                WHERE o.user_id = ? 
                GROUP BY o.id 
                ORDER BY o.created_at DESC
            ");
            $stmt->execute([$user['user_id']]);
            $orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            echo json_encode(['status' => 'success', 'data' => $orders]);
            
        } elseif ($path === 'wallet') {
            // Get wallet balance (Auth required)
            $user = authenticate();
            if (!$user) return;
            
            $stmt = $pdo->prepare("
                SELECT w.*, 
                       (SELECT COUNT(*) FROM wallet_transactions WHERE wallet_id = w.id) as transaction_count
                FROM wallet w 
                WHERE w.user_id = ?
            ");
            $stmt->execute([$user['user_id']]);
            $wallet = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($wallet) {
                // Get recent transactions
                $stmt = $pdo->prepare("
                    SELECT * FROM wallet_transactions 
                    WHERE wallet_id = ? 
                    ORDER BY created_at DESC 
                    LIMIT 10
                ");
                $stmt->execute([$wallet['id']]);
                $transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                $wallet['recent_transactions'] = $transactions;
            }
            
            echo json_encode(['status' => 'success', 'data' => $wallet]);
            
        } else {
            http_response_code(404);
            echo json_encode(['status' => 'error', 'message' => 'Endpoint not found']);
        }
        break;
        
    case 'POST':
        $input = json_decode(file_get_contents('php://input'), true);
        
        if ($path === 'register') {
            // Register user
            $errors = validate_input(['phone', 'password'], $input);
            if (!empty($errors)) {
                http_response_code(400);
                echo json_encode(['status' => 'error', 'message' => 'Validation failed', 'errors' => $errors]);
                return;
            }
            
            // Check if phone exists
            $stmt = $pdo->prepare("SELECT id FROM users WHERE phone = ?");
            $stmt->execute([$input['phone']]);
            if ($stmt->fetch()) {
                http_response_code(409);
                echo json_encode(['status' => 'error', 'message' => 'Phone number already exists']);
                return;
            }
            
            $hashedPassword = password_hash($input['password'], PASSWORD_DEFAULT);
            $roleId = $input['role_id'] ?? 1; // Default to customer
            
            try {
                $pdo->beginTransaction();
                
                // Insert user
                $stmt = $pdo->prepare("INSERT INTO users (phone, email, password_hash, role_id) VALUES (?, ?, ?, ?)");
                $stmt->execute([
                    $input['phone'], 
                    $input['email'] ?? null, 
                    $hashedPassword, 
                    $roleId
                ]);
                
                $userId = $pdo->lastInsertId();
                
                // Create user profile
                if (isset($input['full_name'])) {
                    $stmt = $pdo->prepare("INSERT INTO user_profiles (user_id, full_name) VALUES (?, ?)");
                    $stmt->execute([$userId, $input['full_name']]);
                }
                
                // Create wallet
                $stmt = $pdo->prepare("INSERT INTO wallet (user_id, balance) VALUES (?, 0.00)");
                $stmt->execute([$userId]);
                
                $pdo->commit();
                
                // Generate JWT token
                $token_payload = [
                    'user_id' => $userId,
                    'phone' => $input['phone'],
                    'role_id' => $roleId,
                    'iat' => time(),
                    'exp' => time() + (60 * 60 * 24) // 24 hours
                ];
                $token = generate_jwt($token_payload);
                
                // Get created user
                $stmt = $pdo->prepare("
                    SELECT u.id, u.phone, u.email, u.role_id, u.is_verified, u.status, u.created_at, 
                           up.full_name, r.role_name 
                    FROM users u 
                    LEFT JOIN user_profiles up ON u.id = up.user_id 
                    LEFT JOIN roles r ON u.role_id = r.id 
                    WHERE u.id = ?
                ");
                $stmt->execute([$userId]);
                $user = $stmt->fetch(PDO::FETCH_ASSOC);
                
                http_response_code(201);
                echo json_encode([
                    'status' => 'success', 
                    'message' => 'User registered successfully', 
                    'data' => $user,
                    'token' => $token
                ]);
                
            } catch (Exception $e) {
                $pdo->rollBack();
                http_response_code(500);
                echo json_encode(['status' => 'error', 'message' => 'Registration failed: ' . $e->getMessage()]);
            }
            
        } elseif ($path === 'login') {
            // Login user
            $errors = validate_input(['phone', 'password'], $input);
            if (!empty($errors)) {
                http_response_code(400);
                echo json_encode(['status' => 'error', 'message' => 'Validation failed', 'errors' => $errors]);
                return;
            }
            
            $stmt = $pdo->prepare("
                SELECT u.*, up.full_name, r.role_name 
                FROM users u 
                LEFT JOIN user_profiles up ON u.id = up.user_id 
                LEFT JOIN roles r ON u.role_id = r.id 
                WHERE u.phone = ?
            ");
            $stmt->execute([$input['phone']]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($user && password_verify($input['password'], $user['password_hash'])) {
                if ($user['status'] != 1) {
                    http_response_code(403);
                    echo json_encode(['status' => 'error', 'message' => 'Account is disabled']);
                    return;
                }
                
                unset($user['password_hash']);
                
                // Generate JWT token
                $token_payload = [
                    'user_id' => $user['id'],
                    'phone' => $user['phone'],
                    'role_id' => $user['role_id'],
                    'iat' => time(),
                    'exp' => time() + (60 * 60 * 24) // 24 hours
                ];
                $token = generate_jwt($token_payload);
                
                echo json_encode([
                    'status' => 'success', 
                    'message' => 'Login successful', 
                    'data' => $user,
                    'token' => $token
                ]);
            } else {
                http_response_code(401);
                echo json_encode(['status' => 'error', 'message' => 'Invalid phone or password']);
            }
            
        } elseif ($path === 'verify-otp') {
            // Verify OTP
            $errors = validate_input(['phone', 'otp_code'], $input);
            if (!empty($errors)) {
                http_response_code(400);
                echo json_encode(['status' => 'error', 'message' => 'Validation failed', 'errors' => $errors]);
                return;
            }
            
            // Check OTP in database
            $stmt = $pdo->prepare("
                SELECT * FROM otp_verifications 
                WHERE phone = ? AND otp_code = ? AND is_used = 0 AND expires_at > NOW()
            ");
            $stmt->execute([$input['phone'], $input['otp_code']]);
            $otp = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$otp) {
                http_response_code(400);
                echo json_encode(['status' => 'error', 'message' => 'Invalid or expired OTP']);
                return;
            }
            
            // Mark OTP as used
            $stmt = $pdo->prepare("UPDATE otp_verifications SET is_used = 1 WHERE id = ?");
            $stmt->execute([$otp['id']]);
            
            // Mark user as verified
            $stmt = $pdo->prepare("UPDATE users SET is_verified = 1 WHERE phone = ?");
            $stmt->execute([$input['phone']]);
            
            echo json_encode(['status' => 'success', 'message' => 'OTP verified successfully']);
            
        } elseif ($path === 'products') {
            // Create product (Vendor only)
            $user = authenticate();
            if (!$user) return;
            
            if (!has_role($user, [2, 5])) { // Only vendors and providers
                http_response_code(403);
                echo json_encode(['status' => 'error', 'message' => 'Access denied. Vendor role required.']);
                return;
            }
            
            $errors = validate_input(['title', 'category_id', 'price'], $input);
            if (!empty($errors)) {
                http_response_code(400);
                echo json_encode(['status' => 'error', 'message' => 'Validation failed', 'errors' => $errors]);
                return;
            }
            
            $slug = strtolower(trim(preg_replace('/[^A-Za-z0-9-]+/', '-', $input['title']))) . '-' . time();
            
            $stmt = $pdo->prepare("
                INSERT INTO products (vendor_user_id, category_id, sub_category_id, title, slug, 
                                    short_description, description, price, image_url, features) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $user['user_id'],
                $input['category_id'],
                $input['sub_category_id'] ?? null,
                $input['title'],
                $slug,
                $input['short_description'] ?? null,
                $input['description'] ?? null,
                $input['price'],
                $input['image_url'] ?? null,
                isset($input['features']) ? json_encode($input['features']) : null
            ]);
            
            $product_id = $pdo->lastInsertId();
            
            $stmt = $pdo->prepare("SELECT * FROM products WHERE id = ?");
            $stmt->execute([$product_id]);
            $product = $stmt->fetch(PDO::FETCH_ASSOC);
            
            http_response_code(201);
            echo json_encode([
                'status' => 'success', 
                'message' => 'Product created successfully', 
                'data' => $product
            ]);
            
        } elseif ($path === 'cart') {
            // Add to cart (Auth required)
            $user = authenticate();
            if (!$user) return;
            
            $errors = validate_input(['product_id', 'quantity'], $input);
            if (!empty($errors)) {
                http_response_code(400);
                echo json_encode(['status' => 'error', 'message' => 'Validation failed', 'errors' => $errors]);
                return;
            }
            
            // Get product price
            $stmt = $pdo->prepare("SELECT price FROM products WHERE id = ? AND is_active = 1");
            $stmt->execute([$input['product_id']]);
            $product = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$product) {
                http_response_code(404);
                echo json_encode(['status' => 'error', 'message' => 'Product not found']);
                return;
            }
            
            // Check if item already in cart
            $stmt = $pdo->prepare("SELECT id, quantity FROM cart WHERE user_id = ? AND product_id = ?");
            $stmt->execute([$user['user_id'], $input['product_id']]);
            $existing_item = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($existing_item) {
                // Update quantity
                $new_quantity = $existing_item['quantity'] + $input['quantity'];
                $stmt = $pdo->prepare("UPDATE cart SET quantity = ?, price = ? WHERE id = ?");
                $stmt->execute([$new_quantity, $product['price'], $existing_item['id']]);
            } else {
                // Add new item
                $stmt = $pdo->prepare("
                    INSERT INTO cart (user_id, product_id, quantity, price) 
                    VALUES (?, ?, ?, ?)
                ");
                $stmt->execute([
                    $user['user_id'],
                    $input['product_id'],
                    $input['quantity'],
                    $product['price']
                ]);
            }
            
            echo json_encode(['status' => 'success', 'message' => 'Item added to cart']);
            
        } elseif ($path === 'orders') {
            // Create order (Auth required)
            $user = authenticate();
            if (!$user) return;
            
            // This would typically involve payment processing
            // For now, we'll create a simple order
            
            $order_number = 'ORD' . date('Ymd') . strtoupper(uniqid());
            
            try {
                $pdo->beginTransaction();
                
                // Create order
                $stmt = $pdo->prepare("
                    INSERT INTO orders (created_yyyymm, user_id, order_number, total_amount, final_amount) 
                    VALUES (?, ?, ?, ?, ?)
                ");
                $stmt->execute([
                    date('Ym'),
                    $user['user_id'],
                    $order_number,
                    0, // Will be calculated
                    0  // Will be calculated
                ]);
                
                $order_id = $pdo->lastInsertId();
                
                // Here you would typically add order items from cart
                // For demo, we'll just create a simple order
                
                $pdo->commit();
                
                echo json_encode([
                    'status' => 'success', 
                    'message' => 'Order created successfully', 
                    'order_id' => $order_id,
                    'order_number' => $order_number
                ]);
                
            } catch (Exception $e) {
                $pdo->rollBack();
                http_response_code(500);
                echo json_encode(['status' => 'error', 'message' => 'Order creation failed: ' . $e->getMessage()]);
            }
            
        } else {
            http_response_code(404);
            echo json_encode(['status' => 'error', 'message' => 'Endpoint not found']);
        }
        break;
        
    case 'PUT':
        // For updating resources (you can extend this)
        http_response_code(501);
        echo json_encode(['status' => 'error', 'message' => 'Not implemented']);
        break;
        
    case 'DELETE':
        // For deleting resources (you can extend this)
        http_response_code(501);
        echo json_encode(['status' => 'error', 'message' => 'Not implemented']);
        break;
        
    default:
        http_response_code(405);
        echo json_encode(['status' => 'error', 'message' => 'Method not allowed']);
}
?>