<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Twilio\Rest\Client;
use Twilio\TwiML\VoiceResponse;
use App\Services\AI\Phi3Service;
use App\Services\AI\SimpleResponseService;
use App\Models\CallLog;
use App\Models\Contact;
use Illuminate\Support\Facades\Log;

class TwilioPhoneController extends Controller
{
    protected $twilio;
    protected $aiService;

    public function __construct()
    {
        $this->twilio = new Client(
            config('twilio.account_sid'),
            config('twilio.auth_token')
        );
        $this->aiService = new SimpleResponseService();
    }

    /**
     * Handle incoming calls
     */
    public function handleIncomingCall(Request $request)
    {
        Log::info('Incoming call received', $request->all());

        $response = new VoiceResponse();
        
        // Get caller information
        $from = $request->input('From');
        $to = $request->input('To');
        
        // Log the call
        $this->logCall($from, $to, 'incoming', 'started');
        
        // Greet the caller with AI-generated message
        $greeting = $this->generateAIGreeting($from);
        
        $response->say($greeting, [
            'voice' => config('twilio.voice.voice'),
            'language' => config('twilio.voice.language'),
            'speed' => config('twilio.voice.speed'),
            'pitch' => config('twilio.voice.pitch'),
            'volume' => config('twilio.voice.volume')
        ]);
        
        // Start gathering speech input - ultra-fast timeouts
        $gather = $response->gather([
            'input' => 'speech',
            'action' => route('twilio.speech'),
            'method' => 'POST',
            'timeout' => 3,        // Ultra-fast: 3 seconds
            'speechTimeout' => 1,  // Ultra-fast: 1 second
            'language' => config('twilio.voice.language')
        ]);
        
        $gather->say('How can I help you today?', [
            'voice' => config('twilio.voice.voice'),
            'language' => config('twilio.voice.language'),
            'speed' => config('twilio.voice.speed'),
            'pitch' => config('twilio.voice.pitch'),
            'volume' => config('twilio.voice.volume')
        ]);
        
        // Fallback if no speech detected
        $response->say('I didn\'t hear anything. Please call back if you need assistance.', [
            'voice' => config('twilio.voice.voice'),
            'language' => config('twilio.voice.language'),
            'speed' => config('twilio.voice.speed'),
            'pitch' => config('twilio.voice.pitch'),
            'volume' => config('twilio.voice.volume')
        ]);
        
        return response($response, 200)->header('Content-Type', 'text/xml');
    }

    /**
     * Handle speech input from caller
     */
    public function handleSpeech(Request $request)
    {
        Log::info('Speech received', $request->all());

        $response = new VoiceResponse();
        
        $speechResult = $request->input('SpeechResult');
        $from = $request->input('From');
        $to = $request->input('To');
        
        if ($speechResult) {
            // Process speech with AI
            $aiResponse = $this->processWithAI($speechResult, $from);
            
            $response->say($aiResponse, [
                'voice' => config('twilio.voice.voice'),
                'language' => config('twilio.voice.language'),
                'speed' => config('twilio.voice.speed'),
                'pitch' => config('twilio.voice.pitch'),
                'volume' => config('twilio.voice.volume')
            ]);
            
            // Continue conversation - ultra-fast timeouts
            $gather = $response->gather([
                'input' => 'speech',
                'action' => route('twilio.speech'),
                'method' => 'POST',
                'timeout' => 3,        // Ultra-fast: 3 seconds
                'speechTimeout' => 1,  // Ultra-fast: 1 second
                'language' => config('twilio.voice.language')
            ]);
            
            $gather->say('Is there anything else I can help you with?', [
                'voice' => config('twilio.voice.voice'),
                'language' => config('twilio.voice.language'),
                'speed' => config('twilio.voice.speed'),
                'pitch' => config('twilio.voice.pitch'),
                'volume' => config('twilio.voice.volume')
            ]);
        }
        
        // End call if no response
        $response->say('Thank you for calling. Have a great day!', [
            'voice' => config('twilio.voice.voice'),
            'language' => config('twilio.voice.language'),
            'speed' => config('twilio.voice.speed'),
            'pitch' => config('twilio.voice.pitch'),
            'volume' => config('twilio.voice.volume')
        ]);
        
        $response->hangup();
        
        return response($response, 200)->header('Content-Type', 'text/xml');
    }

    /**
     * Make an outbound call
     */
    public function makeCall(Request $request)
    {
        $request->validate([
            'to' => 'required|string',
            'message' => 'nullable|string'
        ]);

        try {
            $to = $request->input('to');
            $message = $request->input('message', 'Hello! This is DizzCall AI calling you.');
            
            // Log the outbound call
            $this->logCall(config('twilio.phone_number'), $to, 'outbound', 'initiated');
            
            // Use ngrok URL for outbound calls
            $webhookUrl = 'https://d4f73a63faa0.ngrok-free.app/dizzcall-ai/public/api/twilio/webhook/outbound?message=' . urlencode($message);
            
            $call = $this->twilio->calls->create(
                $to, // to
                config('twilio.phone_number'), // from
                [
                    'url' => $webhookUrl,
                    'method' => 'POST',
                    'timeout' => 30,
                    'record' => config('twilio.call.record'),
                    'transcribe' => config('twilio.call.transcribe')
                ]
            );
            
            return response()->json([
                'success' => true,
                'call_sid' => $call->sid,
                'status' => $call->status,
                'message' => 'Call initiated successfully'
            ]);
            
        } catch (\Exception $e) {
            Log::error('Failed to make call: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'error' => 'Failed to make call: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Handle outbound call
     */
    public function handleOutboundCall(Request $request)
    {
        $response = new VoiceResponse();
        
        $message = $request->input('message', 'Hello! This is DizzCall AI calling you.');
        
        $response->say($message, [
            'voice' => config('twilio.voice.voice'),
            'language' => config('twilio.voice.language'),
            'speed' => config('twilio.voice.speed'),
            'pitch' => config('twilio.voice.pitch'),
            'volume' => config('twilio.voice.volume')
        ]);
        
        // Start conversation - ultra-fast timeouts
        $gather = $response->gather([
            'input' => 'speech',
            'action' => route('twilio.speech'),
            'method' => 'POST',
            'timeout' => 3,        // Ultra-fast: 3 seconds
            'speechTimeout' => 1,  // Ultra-fast: 1 second
            'language' => config('twilio.voice.language')
        ]);
        
        $gather->say('How can I assist you today?', [
            'voice' => config('twilio.voice.voice'),
            'language' => config('twilio.voice.language'),
            'speed' => config('twilio.voice.speed'),
            'pitch' => config('twilio.voice.pitch'),
            'volume' => config('twilio.voice.volume')
        ]);
        
        $response->say('Thank you for your time. Goodbye!', [
            'voice' => config('twilio.voice.voice'),
            'language' => config('twilio.voice.language'),
            'speed' => config('twilio.voice.speed'),
            'pitch' => config('twilio.voice.pitch'),
            'volume' => config('twilio.voice.volume')
        ]);
        
        $response->hangup();
        
        return response($response, 200)->header('Content-Type', 'text/xml');
    }

    /**
     * Get call status
     */
    public function getCallStatus(Request $request)
    {
        $callSid = $request->input('call_sid');
        
        try {
            $call = $this->twilio->calls($callSid)->fetch();
            
            return response()->json([
                'success' => true,
                'status' => $call->status,
                'duration' => $call->duration,
                'direction' => $call->direction,
                'from' => $call->from,
                'to' => $call->to
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Failed to get call status: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Generate AI greeting for caller
     */
    private function generateAIGreeting($phoneNumber)
    {
        try {
            $contact = Contact::where('phone', $phoneNumber)->first();
            $name = $contact ? $contact->name : 'valued customer';
            
            // Use a simple, fast greeting instead of AI for now
            $greetings = [
                "Hello {$name}! This is Alex from Dizaart Systems. How can I help you today?",
                "Hi there! Alex from Dizaart Systems here. What can I do for you?",
                "Good day! This is Alex from Dizaart Systems. How may I assist you?",
                "Hello! Alex from Dizaart Systems. How can I help you today?"
            ];
            
            return $greetings[array_rand($greetings)];
            
        } catch (\Exception $e) {
            Log::error('Failed to generate greeting: ' . $e->getMessage());
        }
        
        return "Hello! This is Alex from Dizaart Systems. How can I help you today?";
    }

    /**
     * Process speech with AI - Ultra-fast for phone calls
     */
    private function processWithAI($speech, $phoneNumber)
    {
        try {
            // Use simple response service for reliable responses
            $aiResponse = $this->aiService->generateResponse($speech, [], 'phone');
            
            if ($aiResponse['success']) {
                // Log the conversation
                $this->logConversation($phoneNumber, $speech, $aiResponse['response']);
                return $aiResponse['response'];
            }
            
        } catch (\Exception $e) {
            Log::error('Failed to process speech: ' . $e->getMessage());
        }
        
        // Fallback to instant response
        return "Thanks for calling Dizaart Systems! We provide custom software, mobile apps, and AI solutions. How can I assist you?";
    }

    /**
     * Log call information
     */
    private function logCall($from, $to, $direction, $status)
    {
        try {
            // Ensure we have valid phone numbers
            $from = $from ?: 'unknown';
            $to = $to ?: 'unknown';
            
            CallLog::create([
                'from_number' => $from,
                'to_number' => $to,
                'direction' => $direction,
                'status' => $status,
                'started_at' => now(),
                'user_id' => auth()->id() ?? null
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to log call: ' . $e->getMessage());
        }
    }

    /**
     * Log conversation
     */
    private function logConversation($phoneNumber, $userMessage, $aiResponse)
    {
        try {
            // You can create a conversation log table if needed
            Log::info('Conversation logged', [
                'phone' => $phoneNumber,
                'user_message' => $userMessage,
                'ai_response' => $aiResponse,
                'timestamp' => now()
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to log conversation: ' . $e->getMessage());
        }
    }
}
