<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Twilio\Rest\Client;
use App\Services\AI\Phi3Service;

class TestTwilioIntegration extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'twilio:test';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Test Twilio integration and AI voice capabilities';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🧪 Testing Twilio Integration...');
        
        // Test 1: Check Twilio Configuration
        $this->testTwilioConfig();
        
        // Test 2: Test AI Service
        $this->testAIService();
        
        // Test 3: Test Database Connection
        $this->testDatabase();
        
        $this->info('✅ All tests completed!');
    }
    
    private function testTwilioConfig()
    {
        $this->info('📞 Testing Twilio Configuration...');
        
        $accountSid = config('twilio.account_sid');
        $authToken = config('twilio.auth_token');
        $phoneNumber = config('twilio.phone_number');
        
        if (empty($accountSid) || empty($authToken) || empty($phoneNumber)) {
            $this->error('❌ Twilio configuration is incomplete!');
            $this->line('Please set the following in your .env file:');
            $this->line('TWILIO_ACCOUNT_SID=your_account_sid');
            $this->line('TWILIO_AUTH_TOKEN=your_auth_token');
            $this->line('TWILIO_PHONE_NUMBER=+1234567890');
            return false;
        }
        
        try {
            $client = new Client($accountSid, $authToken);
            $account = $client->api->accounts($accountSid)->fetch();
            
            $this->info("✅ Twilio connection successful!");
            $this->line("Account: {$account->friendlyName}");
            $this->line("Phone: {$phoneNumber}");
            
        } catch (\Exception $e) {
            $this->error("❌ Twilio connection failed: {$e->getMessage()}");
            return false;
        }
        
        return true;
    }
    
    private function testAIService()
    {
        $this->info('🤖 Testing AI Service...');
        
        try {
            $aiService = new Phi3Service();
            $response = $aiService->generateResponse('Hello, this is a test message.', [], 'general');
            
            if ($response['success']) {
                $this->info('✅ AI Service is working!');
                $this->line("Response: {$response['response']}");
            } else {
                $this->error('❌ AI Service failed to generate response');
            }
            
        } catch (\Exception $e) {
            $this->error("❌ AI Service error: {$e->getMessage()}");
        }
    }
    
    private function testDatabase()
    {
        $this->info('🗄️ Testing Database...');
        
        try {
            $callLogCount = \App\Models\CallLog::count();
            $this->info("✅ Database connection successful!");
            $this->line("Call logs in database: {$callLogCount}");
            
        } catch (\Exception $e) {
            $this->error("❌ Database error: {$e->getMessage()}");
        }
    }
}
