<?php

namespace App\Http\Controllers\FrontUser;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Call;
use App\Models\WalletTransaction;

class SearchController extends Controller
{
    public function index(Request $request)
    {
        $user = Auth::user();
        $query = $request->get('q', '');
        $type = $request->get('type', 'all'); // all, calls, transactions
        
        $results = [
            'calls' => collect(),
            'transactions' => collect(),
            'total' => 0
        ];

        if (!empty($query)) {
            // Search calls
            if ($type === 'all' || $type === 'calls') {
                $results['calls'] = Call::where('user_id', $user->id)
                    ->where(function($q) use ($query) {
                        $q->where('phone_number', 'like', "%{$query}%")
                          ->orWhere('status', 'like', "%{$query}%")
                          ->orWhere('package_name', 'like', "%{$query}%");
                    })
                    ->orderBy('created_at', 'desc')
                    ->limit(10)
                    ->get();
            }

            // Search wallet transactions
            if ($type === 'all' || $type === 'transactions') {
                $results['transactions'] = WalletTransaction::where('user_id', $user->id)
                    ->where(function($q) use ($query) {
                        $q->where('description', 'like', "%{$query}%")
                          ->orWhere('type', 'like', "%{$query}%")
                          ->orWhere('status', 'like', "%{$query}%");
                    })
                    ->orderBy('created_at', 'desc')
                    ->limit(10)
                    ->get();
            }

            $results['total'] = $results['calls']->count() + $results['transactions']->count();
        }

        return view('FrontUser.search.index', compact('results', 'query', 'type'));
    }

    public function search(Request $request)
    {
        $user = Auth::user();
        $query = $request->get('q', '');
        $type = $request->get('type', 'all');
        
        $results = [];

        if (!empty($query)) {
            switch ($type) {
                case 'calls':
                    $results = Call::where('user_id', $user->id)
                        ->where(function($q) use ($query) {
                            $q->where('phone_number', 'like', "%{$query}%")
                              ->orWhere('status', 'like', "%{$query}%")
                              ->orWhere('package_name', 'like', "%{$query}%");
                        })
                        ->orderBy('created_at', 'desc')
                        ->limit(20)
                        ->get()
                        ->map(function($call) {
                            return [
                                'id' => $call->id,
                                'title' => "Call to {$call->phone_number}",
                                'description' => "Package: {$call->package_name} | Status: {$call->status}",
                                'date' => $call->created_at->format('M d, Y H:i'),
                                'type' => 'call',
                                'url' => route('user.calls.show', $call->id)
                            ];
                        });
                    break;

                case 'transactions':
                    $results = WalletTransaction::where('user_id', $user->id)
                        ->where(function($q) use ($query) {
                            $q->where('description', 'like', "%{$query}%")
                              ->orWhere('type', 'like', "%{$query}%")
                              ->orWhere('status', 'like', "%{$query}%");
                        })
                        ->orderBy('created_at', 'desc')
                        ->limit(20)
                        ->get()
                        ->map(function($transaction) {
                            return [
                                'id' => $transaction->id,
                                'title' => $transaction->description,
                                'description' => "Amount: ₹{$transaction->amount} | Type: {$transaction->type}",
                                'date' => $transaction->created_at->format('M d, Y H:i'),
                                'type' => 'transaction',
                                'url' => route('user.wallet.transactions')
                            ];
                        });
                    break;

                default:
                    $calls = Call::where('user_id', $user->id)
                        ->where(function($q) use ($query) {
                            $q->where('phone_number', 'like', "%{$query}%")
                              ->orWhere('status', 'like', "%{$query}%")
                              ->orWhere('package_name', 'like', "%{$query}%");
                        })
                        ->orderBy('created_at', 'desc')
                        ->limit(10)
                        ->get()
                        ->map(function($call) {
                            return [
                                'id' => $call->id,
                                'title' => "Call to {$call->phone_number}",
                                'description' => "Package: {$call->package_name} | Status: {$call->status}",
                                'date' => $call->created_at->format('M d, Y H:i'),
                                'type' => 'call',
                                'url' => route('user.calls.show', $call->id)
                            ];
                        });

                    $transactions = WalletTransaction::where('user_id', $user->id)
                        ->where(function($q) use ($query) {
                            $q->where('description', 'like', "%{$query}%")
                              ->orWhere('type', 'like', "%{$query}%")
                              ->orWhere('status', 'like', "%{$query}%");
                        })
                        ->orderBy('created_at', 'desc')
                        ->limit(10)
                        ->get()
                        ->map(function($transaction) {
                            return [
                                'id' => $transaction->id,
                                'title' => $transaction->description,
                                'description' => "Amount: ₹{$transaction->amount} | Type: {$transaction->type}",
                                'date' => $transaction->created_at->format('M d, Y H:i'),
                                'type' => 'transaction',
                                'url' => route('user.wallet.transactions')
                            ];
                        });

                    $results = $calls->concat($transactions)->sortByDesc('date')->values();
                    break;
            }
        }

        return response()->json([
            'success' => true,
            'results' => $results,
            'total' => count($results)
        ]);
    }
}