<?php

namespace App\Http\Controllers\FrontUser;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class UserSettingsController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        return view('FrontUser.settings.index', compact('user'));
    }

    public function updateProfile(Request $request)
    {
        Log::info('Updating profile for user: ' . Auth::id(), $request->all());
        try {
            $user = Auth::user();
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'email' => 'required|email|unique:users,email,' . $user->id,
                'contact_number' => 'nullable|string|max:20',
                'company_name' => 'nullable|string|max:255',
                'address' => 'nullable|string',
                'website' => 'nullable|url',
                'theme' => 'nullable|in:light,dark',
                'avatar' => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
            ]);

            if ($request->hasFile('avatar')) {
                Log::info('Avatar upload detected.');
                // Delete old avatar if it exists
                if ($user->avatar) {
                    Log::info('Deleting old avatar: ' . $user->avatar);
                    Storage::delete('public/avatars/' . $user->avatar);
                }

                // Store new avatar and get the filename
                $path = $request->file('avatar')->store('public/avatars');
                $filename = basename($path);
                Log::info('New avatar stored at: ' . $path);

                // Save the new filename to the user
                $validated['avatar'] = $filename;
            }

            $user->update($validated);
            Log::info('Profile updated successfully for user: ' . Auth::id());
            return back()->with('status', 'Profile updated');
        } catch (\Exception $e) {
            Log::error('Error updating profile: ' . $e->getMessage());
            return back()->with('error', 'There was an error updating your profile.');
        }
    }

    public function updateNotifications(Request $request)
    {
        // Persist notification preferences if you have a settings table; placeholder
        return back()->with('status', 'Preferences saved');
    }

    public function updateApi(Request $request)
    {
        // Persist webhook/api key if applicable; placeholder
        return back()->with('status', 'API settings saved');
    }

    public function uploadAvatar(Request $request)
    {
        $request->validate([
            'avatar' => ['required', 'image', 'mimes:jpg,jpeg,png', 'max:2048']
        ]);

        $user = Auth::user();

        // Delete old avatar if it exists
        if ($user->avatar) {
            Storage::delete('public/avatars/' . $user->avatar);
        }

        // Store new avatar and get the filename
        $path = $request->file('avatar')->store('public/avatars');
        $filename = basename($path);

        // Save the new filename to the user
        $user->avatar = $filename;
        $user->save();

        return response()->json([
            'success' => true,
            'avatar_url' => asset('storage/avatars/' . $filename)
        ]);
    }

    public function update(Request $request)
    {
        // Update settings logic
    }
}