<?php

namespace App\Http\Controllers;

use App\Models\Ticket;
use App\Models\TicketReply;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class TicketController extends Controller
{
    public function index()
    {
        if (Auth::user()->is_admin) {
            $tickets = Ticket::latest()->get();
            return view('admin.tickets.index', compact('tickets'));
        }
        $tickets = Ticket::where('user_id', Auth::id())->latest()->get();
        return view('tickets.index', compact('tickets'));
    }

    public function create()
    {
        $user = Auth::user();
        return view('tickets.create', compact('user'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'subject' => 'required|string|max:255',
            'message' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);

        $imagePath = null;
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('tickets', 'public');
        }

        Ticket::create([
            'user_id' => Auth::id(),
            'subject' => $request->subject,
            'message' => $request->message,
            'image_path' => $imagePath,
        ]);

        return redirect()->route('user.tickets.index')->with('success', 'Ticket created successfully.');
    }

    public function show(Ticket $ticket)
    {
        if (Auth::user()->is_admin || $ticket->user_id === Auth::id()) {
            $ticket->load('replies.user');
            if (Auth::user()->is_admin) {
                return view('admin.tickets.show', compact('ticket'));
            }
            return view('tickets.show', compact('ticket'));
        }
        abort(403);
    }

    public function reply(Request $request, Ticket $ticket)
    {
        if ($ticket->user_id !== Auth::id()) {
            abort(403);
        }

        $request->validate([
            'message' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);

        $imagePath = null;
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('ticket_replies', 'public');
        }

        TicketReply::create([
            'ticket_id' => $ticket->id,
            'user_id' => Auth::id(),
            'message' => $request->message,
            'image_path' => $imagePath,
        ]);

        $ticket->update(['status' => 'open']);

        return redirect()->route('user.tickets.show', $ticket)->with('success', 'Reply sent successfully.');
    }

    public function adminReply(Request $request, Ticket $ticket)
    {
        if ($ticket->status == 'closed') {
            return back()->with('error', 'Cannot reply to a closed ticket.');
        }
        $request->validate([
            'message' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);

        $imagePath = null;
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('ticket_replies', 'public');
        }

        TicketReply::create([
            'ticket_id' => $ticket->id,
            'user_id' => Auth::id(),
            'message' => $request->message,
            'image_path' => $imagePath,
        ]);

        $ticket->update(['status' => 'answered']);

        return redirect()->route('admin.tickets.show', $ticket)->with('success', 'Reply sent successfully.');
    }

    public function close(Ticket $ticket)
    {
        $ticket->update(['status' => 'closed']);
        return redirect()->route('admin.tickets.index')->with('success', 'Ticket closed successfully.');
    }
}
