<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Twilio\Rest\Client;
use Twilio\TwiML\VoiceResponse;
use App\Services\AI\Phi3Service;
use App\Models\CallLog;
use App\Models\Contact;
use Illuminate\Support\Facades\Log;

class TwilioPhoneController extends Controller
{
    protected $twilio;
    protected $aiService;

    public function __construct()
    {
        $this->twilio = new Client(
            config('twilio.account_sid'),
            config('twilio.auth_token')
        );
        $this->aiService = new Phi3Service();
    }

    /**
     * Handle incoming calls
     */
    public function handleIncomingCall(Request $request)
    {
        Log::info('Incoming call received', $request->all());

        $response = new VoiceResponse();
        
        // Get caller information
        $from = $request->input('From');
        $to = $request->input('To');
        
        // Log the call
        $this->logCall($from, $to, 'incoming', 'started');
        
        // Greet the caller with AI-generated message
        $greeting = $this->generateAIGreeting($from);
        
        $response->say($greeting, [
            'voice' => config('twilio.voice.voice'),
            'language' => config('twilio.voice.language')
        ]);
        
        // Start gathering speech input
        $gather = $response->gather([
            'input' => 'speech',
            'action' => route('twilio.speech'),
            'method' => 'POST',
            'timeout' => 10,
            'speechTimeout' => 'auto',
            'language' => config('twilio.voice.language')
        ]);
        
        $gather->say('How can I help you today?', [
            'voice' => config('twilio.voice.voice'),
            'language' => config('twilio.voice.language')
        ]);
        
        // Fallback if no speech detected
        $response->say('I didn\'t hear anything. Please call back if you need assistance.', [
            'voice' => config('twilio.voice.voice'),
            'language' => config('twilio.voice.language')
        ]);
        
        return response($response, 200)->header('Content-Type', 'text/xml');
    }

    /**
     * Handle speech input from caller
     */
    public function handleSpeech(Request $request)
    {
        Log::info('Speech received', $request->all());

        $response = new VoiceResponse();
        
        $speechResult = $request->input('SpeechResult');
        $from = $request->input('From');
        $to = $request->input('To');
        
        if ($speechResult) {
            // Process speech with AI
            $aiResponse = $this->processWithAI($speechResult, $from);
            
            $response->say($aiResponse, [
                'voice' => config('twilio.voice.voice'),
                'language' => config('twilio.voice.language')
            ]);
            
            // Continue conversation
            $gather = $response->gather([
                'input' => 'speech',
                'action' => route('twilio.speech'),
                'method' => 'POST',
                'timeout' => 10,
                'speechTimeout' => 'auto',
                'language' => config('twilio.voice.language')
            ]);
            
            $gather->say('Is there anything else I can help you with?', [
                'voice' => config('twilio.voice.voice'),
                'language' => config('twilio.voice.language')
            ]);
        }
        
        // End call if no response
        $response->say('Thank you for calling. Have a great day!', [
            'voice' => config('twilio.voice.voice'),
            'language' => config('twilio.voice.language')
        ]);
        
        $response->hangup();
        
        return response($response, 200)->header('Content-Type', 'text/xml');
    }

    /**
     * Make an outbound call
     */
    public function makeCall(Request $request)
    {
        $request->validate([
            'to' => 'required|string',
            'message' => 'nullable|string'
        ]);

        try {
            $to = $request->input('to');
            $message = $request->input('message', 'Hello! This is DizzCall AI calling you.');
            
            // Log the outbound call
            $this->logCall(config('twilio.phone_number'), $to, 'outbound', 'initiated');
            
            $call = $this->twilio->calls->create(
                $to, // to
                config('twilio.phone_number'), // from
                [
                    'url' => route('twilio.outbound', ['message' => urlencode($message)]),
                    'method' => 'POST',
                    'timeout' => 30,
                    'record' => config('twilio.call.record'),
                    'transcribe' => config('twilio.call.transcribe')
                ]
            );
            
            return response()->json([
                'success' => true,
                'call_sid' => $call->sid,
                'status' => $call->status,
                'message' => 'Call initiated successfully'
            ]);
            
        } catch (\Exception $e) {
            Log::error('Failed to make call: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'error' => 'Failed to make call: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Handle outbound call
     */
    public function handleOutboundCall(Request $request)
    {
        $response = new VoiceResponse();
        
        $message = $request->input('message', 'Hello! This is DizzCall AI calling you.');
        
        $response->say($message, [
            'voice' => config('twilio.voice.voice'),
            'language' => config('twilio.voice.language')
        ]);
        
        // Start conversation
        $gather = $response->gather([
            'input' => 'speech',
            'action' => route('twilio.speech'),
            'method' => 'POST',
            'timeout' => 10,
            'speechTimeout' => 'auto',
            'language' => config('twilio.voice.language')
        ]);
        
        $gather->say('How can I assist you today?', [
            'voice' => config('twilio.voice.voice'),
            'language' => config('twilio.voice.language')
        ]);
        
        $response->say('Thank you for your time. Goodbye!', [
            'voice' => config('twilio.voice.voice'),
            'language' => config('twilio.voice.language')
        ]);
        
        $response->hangup();
        
        return response($response, 200)->header('Content-Type', 'text/xml');
    }

    /**
     * Get call status
     */
    public function getCallStatus(Request $request)
    {
        $callSid = $request->input('call_sid');
        
        try {
            $call = $this->twilio->calls($callSid)->fetch();
            
            return response()->json([
                'success' => true,
                'status' => $call->status,
                'duration' => $call->duration,
                'direction' => $call->direction,
                'from' => $call->from,
                'to' => $call->to
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Failed to get call status: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Generate AI greeting for caller
     */
    private function generateAIGreeting($phoneNumber)
    {
        try {
            $contact = Contact::where('phone', $phoneNumber)->first();
            $name = $contact ? $contact->name : 'valued customer';
            
            $prompt = "Generate a warm, professional greeting for a phone call. The caller's name is {$name}. Keep it under 20 words.";
            
            $aiResponse = $this->aiService->generateResponse($prompt, [], 'general');
            
            if ($aiResponse['success']) {
                return $aiResponse['response'];
            }
            
        } catch (\Exception $e) {
            Log::error('Failed to generate AI greeting: ' . $e->getMessage());
        }
        
        return "Hello! Thank you for calling DizzCall AI. I'm here to help you today.";
    }

    /**
     * Process speech with AI
     */
    private function processWithAI($speech, $phoneNumber)
    {
        try {
            // Get contact context
            $contact = Contact::where('phone', $phoneNumber)->first();
            $context = $contact ? ['contact_name' => $contact->name, 'contact_phone' => $phoneNumber] : [];
            
            $aiResponse = $this->aiService->generateResponse($speech, $context, 'general');
            
            if ($aiResponse['success']) {
                // Log the conversation
                $this->logConversation($phoneNumber, $speech, $aiResponse['response']);
                return $aiResponse['response'];
            }
            
        } catch (\Exception $e) {
            Log::error('Failed to process speech with AI: ' . $e->getMessage());
        }
        
        return "I apologize, but I'm having trouble processing your request right now. Could you please try again?";
    }

    /**
     * Log call information
     */
    private function logCall($from, $to, $direction, $status)
    {
        try {
            CallLog::create([
                'from_number' => $from,
                'to_number' => $to,
                'direction' => $direction,
                'status' => $status,
                'started_at' => now(),
                'user_id' => auth()->id() ?? null
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to log call: ' . $e->getMessage());
        }
    }

    /**
     * Log conversation
     */
    private function logConversation($phoneNumber, $userMessage, $aiResponse)
    {
        try {
            // You can create a conversation log table if needed
            Log::info('Conversation logged', [
                'phone' => $phoneNumber,
                'user_message' => $userMessage,
                'ai_response' => $aiResponse,
                'timestamp' => now()
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to log conversation: ' . $e->getMessage());
        }
    }
}
