<?php

namespace App\Http\Controllers;

use App\Models\WalletTransaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Razorpay\Api\Api;
use Illuminate\Support\Facades\Mail;
use App\Mail\WalletRechargeMail;

class WalletController extends Controller
{
    public function showTopUpForm()
    {
        $user = Auth::user();
        return view('wallet.topup', compact('user'));
    }

    public function createOrder(Request $request)
    {
        $request->validate([
            'amount' => ['required', 'numeric', 'min:1', 'max:1000000'],
        ]);

        $user = Auth::user();

        $amountInPaise = (int) round($request->input('amount') * 100);

        $api = new Api(config('services.razorpay.key_id'), config('services.razorpay.key_secret'));

        $order = $api->order->create([
            'amount' => $amountInPaise,
            'currency' => 'INR',
            'receipt' => 'wallet_' . $user->id . '_' . now()->timestamp,
            'payment_capture' => 1,
            'notes' => [
                'user_id' => (string) $user->id,
                'purpose' => 'wallet_topup',
            ],
        ]);

        $txn = WalletTransaction::create([
            'user_id' => $user->id,
            'type' => 'credit',
            'amount' => $request->input('amount'),
            'currency' => 'INR',
            'status' => 'pending',
            'razorpay_order_id' => $order['id'] ?? null,
            'meta' => [
                'receipt' => $order['receipt'] ?? null,
            ],
        ]);

        return response()->json([
            'order_id' => $order['id'],
            'key' => config('services.razorpay.key_id'),
            'amount' => $amountInPaise,
            'currency' => 'INR',
            'txn_id' => $txn->id,
        ]);
    }

    public function verify(Request $request)
    {
        $request->validate([
            'razorpay_payment_id' => 'required|string',
            'razorpay_order_id' => 'required|string',
            'razorpay_signature' => 'required|string',
            'txn_id' => 'required|integer',
        ]);

        $user = Auth::user();

        $generatedSignature = hash_hmac(
            'sha256',
            $request->razorpay_order_id . '|' . $request->razorpay_payment_id,
            config('services.razorpay.key_secret')
        );

        if (!hash_equals($generatedSignature, $request->razorpay_signature)) {
            return response()->json(['message' => 'Invalid signature'], 422);
        }

        DB::transaction(function () use ($request, $user) {
            $txn = WalletTransaction::where('id', $request->txn_id)
                ->where('user_id', $user->id)
                ->lockForUpdate()
                ->firstOrFail();

            if ($txn->status === 'success') {
                return;
            }

            $txn->update([
                'status' => 'success',
                'razorpay_payment_id' => $request->razorpay_payment_id,
                'razorpay_signature' => $request->razorpay_signature,
            ]);

            $user->refresh();
            $user->wallet_balance = ($user->wallet_balance ?? 0) + $txn->amount;
            $user->save();

            $txn->update([
                'balance_after' => $user->wallet_balance,
            ]);

            try {
                Mail::to($user->email)->send(new WalletRechargeMail(
                    $user->name,
                    (float) $txn->amount,
                    (float) $user->wallet_balance,
                    $txn->razorpay_payment_id
                ));
            } catch (\Throwable $e) {
                // ignore mail failure
            }
        });

        return response()->json(['message' => 'Payment verified and wallet updated']);
    }
}


