<?php
defined('BASEPATH') OR exit('No direct script access allowed');

// Extract model data
$model_name = !empty($model['model_name']) ? $model['model_name'] : 
              (!empty($model['original_name']) ? pathinfo($model['original_name'], PATHINFO_FILENAME) : 'Model Portfolio');

// Clean model name
$model_name = str_replace(['-', '_', '.pdf'], ' ', $model_name);
$model_name = ucwords(strtolower(trim($model_name)));

// Get measurements
function getMeasurement($model, $key) {
    $fields = [
        'height' => ['model_height', 'height', 'Height'],
        'chest' => ['model_chest', 'chest', 'chest_size', 'Chest'],
        'waist' => ['model_waist', 'waist', 'waist_size', 'Waist'],
        'hips' => ['model_hips', 'hips', 'hips_size', 'Hips'],
        'shoe' => ['model_shoe_size', 'shoe_size', 'shoe', 'Shoe'],
        'eyes' => ['eye_color', 'eyes', 'Eyes'],
        'hair' => ['hair_color', 'hair', 'Hair']
    ];
    
    if (isset($fields[$key])) {
        foreach ($fields[$key] as $field) {
            if (!empty($model[$field])) {
                return $model[$field];
            }
        }
    }
    
    return '';
}

// Instagram
$instagram_handle = $model['instagram_handle'] ?? '';
$instagram_url = $model['instagram_url'] ?? '';
if ($instagram_handle && !$instagram_url && strpos($instagram_handle, '@') === 0) {
    $instagram_url = 'https://instagram.com/' . substr($instagram_handle, 1);
}

// Contact info
$contact_info = $model['contact_info'] ?? '';
$extracted_details = $model['extracted_details'] ?? $model['extracted_text'] ?? '';

// Split images into rows (3 per row for desktop)
$image_rows = array_chunk($images, 3);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($page_title); ?></title>
    <meta name="description" content="Professional model portfolio - <?php echo htmlspecialchars($model_name); ?> - GO Productions">
    
    <!-- Fonts -->
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Bebas+Neue&family=Montserrat:wght@300;400;500;600;700&family=Inter:wght@300;400;500&display=swap" rel="stylesheet">
    
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    
    <style>
        /* RESET & BASE */
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        html {
            scroll-behavior: smooth;
        }

        body {
            font-family: 'Montserrat', sans-serif;
            background: #0a0a0a;
            color: #ffffff;
            line-height: 1.6;
            overflow-x: hidden;
        }

        a {
            text-decoration: none;
            color: inherit;
            transition: all 0.3s ease;
        }

        img {
            max-width: 100%;
            height: auto;
            display: block;
        }

        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 0 20px;
        }

        /* HEADER SECTION */
        .portfolio-header {
            background: linear-gradient(135deg, #111111 0%, #1a1a1a 100%);
            padding: 40px 0;
            border-bottom: 1px solid #333;
            position: relative;
            overflow: hidden;
        }

        .header-content {
            display: flex;
            align-items: center;
            justify-content: space-between;
            position: relative;
            z-index: 2;
        }

        .logo-container {
            flex-shrink: 0;
        }

        .logo {
            height: 60px;
            width: auto;
            filter: brightness(0) invert(1);
        }

        .model-name-container {
            flex-grow: 1;
            text-align: center;
            padding: 0 40px;
        }

        .model-name {
            font-family: 'Bebas Neue', sans-serif;
            font-size: 72px;
            font-weight: 400;
            letter-spacing: 3px;
            line-height: 1;
            background: linear-gradient(90deg, #ffffff, #cccccc, #ffffff);
            background-size: 200% auto;
            background-clip: text;
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            animation: shine 3s linear infinite;
            text-transform: uppercase;
        }

        @keyframes shine {
            0% { background-position: 200% center; }
            100% { background-position: -200% center; }
        }

        /* MODEL INFO SECTION */
        .model-info-section {
            padding: 60px 0;
            background: #111111;
        }

        .model-info-grid {
            display: grid;
            grid-template-columns: 300px 1fr;
            gap: 60px;
            align-items: start;
        }

        @media (max-width: 992px) {
            .model-info-grid {
                grid-template-columns: 1fr;
                gap: 40px;
            }
        }

        /* LEFT SIDE - DETAILS */
        .model-details {
            background: #1a1a1a;
            padding: 30px;
            border-radius: 10px;
            border: 1px solid #333;
            position: sticky;
            top: 20px;
        }

        .details-header {
            margin-bottom: 25px;
            padding-bottom: 15px;
            border-bottom: 2px solid #444;
        }

        .details-title {
            font-family: 'Bebas Neue', sans-serif;
            font-size: 32px;
            letter-spacing: 1px;
            color: #ffffff;
            text-transform: uppercase;
        }

        .measurements-list {
            list-style: none;
            margin-bottom: 30px;
        }

        .measurement-item {
            display: flex;
            justify-content: space-between;
            padding: 12px 0;
            border-bottom: 1px solid #333;
        }

        .measurement-item:last-child {
            border-bottom: none;
        }

        .measurement-label {
            color: #999;
            font-weight: 500;
            font-size: 14px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .measurement-value {
            color: #ffffff;
            font-weight: 600;
            font-size: 16px;
        }

        /* Contact Info */
        .contact-info {
            margin-top: 25px;
            padding-top: 25px;
            border-top: 1px solid #333;
        }

        .contact-item {
            display: flex;
            align-items: center;
            gap: 12px;
            margin-bottom: 15px;
            padding: 10px;
            background: #222;
            border-radius: 6px;
            transition: background 0.3s ease;
        }

        .contact-item:hover {
            background: #2a2a2a;
        }

        .contact-icon {
            width: 36px;
            height: 36px;
            background: #333;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 16px;
            color: #fff;
        }

        .contact-text {
            flex-grow: 1;
        }

        .contact-label {
            font-size: 11px;
            color: #999;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            margin-bottom: 2px;
        }

        .contact-value {
            font-size: 14px;
            font-weight: 500;
            color: #fff;
            word-break: break-all;
        }

        /* RIGHT SIDE - GALLERY */
        .gallery-section {
            width: 100%;
        }

        .gallery-title {
            font-family: 'Bebas Neue', sans-serif;
            font-size: 42px;
            letter-spacing: 2px;
            margin-bottom: 30px;
            color: #ffffff;
            text-align: center;
            text-transform: uppercase;
        }

        .image-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 20px;
        }

        .image-item {
            position: relative;
            border-radius: 8px;
            overflow: hidden;
            aspect-ratio: 3/4;
            transition: transform 0.3s ease;
        }

        .image-item:hover {
            transform: translateY(-5px);
        }

        .image-item img {
            width: 100%;
            height: 100%;
            object-fit: cover;
            transition: transform 0.5s ease;
        }

        .image-item:hover img {
            transform: scale(1.05);
        }

        .image-overlay {
            position: absolute;
            bottom: 0;
            left: 0;
            right: 0;
            background: linear-gradient(to top, rgba(0,0,0,0.8), transparent);
            padding: 20px;
            opacity: 0;
            transition: opacity 0.3s ease;
        }

        .image-item:hover .image-overlay {
            opacity: 1;
        }

        .image-number {
            font-size: 14px;
            color: #fff;
            font-weight: 500;
        }

        /* FOOTER */
        .portfolio-footer {
            background: #000000;
            padding: 40px 0;
            text-align: center;
            border-top: 1px solid #333;
            margin-top: 60px;
        }

        .footer-logo {
            height: 40px;
            width: auto;
            margin-bottom: 20px;
            filter: brightness(0) invert(1);
            opacity: 0.7;
        }

        .footer-text {
            color: #666;
            font-size: 14px;
            font-weight: 400;
            letter-spacing: 1px;
        }

        .copyright {
            margin-top: 10px;
            font-size: 12px;
            color: #444;
        }

        /* ACTION BUTTONS */
        .action-buttons {
            position: fixed;
            bottom: 30px;
            right: 30px;
            z-index: 1000;
            display: flex;
            gap: 10px;
        }

        .action-btn {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            background: #111;
            border: 2px solid #333;
            color: #fff;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 18px;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .action-btn:hover {
            background: #222;
            border-color: #555;
            transform: translateY(-3px);
        }

        /* RESPONSIVE DESIGN */
        @media (max-width: 768px) {
            .header-content {
                flex-direction: column;
                gap: 20px;
            }

            .logo-container {
                order: 1;
            }

            .model-name-container {
                order: 2;
                padding: 0;
            }

            .model-name {
                font-size: 48px;
                letter-spacing: 2px;
            }

            .model-info-grid {
                gap: 30px;
            }

            .model-details {
                position: static;
            }

            .gallery-title {
                font-size: 32px;
            }

            .image-grid {
                grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
            }

            .action-buttons {
                bottom: 20px;
                right: 20px;
            }
        }

        @media (max-width: 480px) {
            .model-name {
                font-size: 36px;
            }

            .details-title {
                font-size: 24px;
            }

            .gallery-title {
                font-size: 28px;
            }

            .image-grid {
                grid-template-columns: 1fr;
            }
        }

        /* PRINT STYLES */
        @media print {
            .action-buttons {
                display: none;
            }
            
            body {
                background: white;
                color: black;
            }
            
            .portfolio-header {
                background: white;
                border-bottom: 2px solid #000;
            }
            
            .model-name {
                background: none;
                -webkit-text-fill-color: black;
                animation: none;
            }
            
            .logo {
                filter: none;
            }
            
            .model-details {
                background: #f5f5f5;
                border: 1px solid #ddd;
            }
            
            .contact-item {
                background: #eee;
            }
            
            .portfolio-footer {
                background: white;
                border-top: 2px solid #000;
            }
            
            .footer-logo {
                filter: none;
                opacity: 1;
            }
        }
    </style>
</head>
<body>
    <!-- HEADER SECTION -->
    <header class="portfolio-header">
        <div class="container">
            <div class="header-content">
                <!-- Logo on Left -->
                <div class="logo-container">
                    <img src="<?php echo base_url('assets/images/logogo.png'); ?>" 
                         alt="GO Productions" 
                         class="logo">
                </div>
                
                <!-- Model Name in Center with Running Effect -->
                <div class="model-name-container">
                    <h1 class="model-name"><?php echo htmlspecialchars($model_name); ?></h1>
                </div>
                
                <!-- Right Side (Empty for balance) -->
                <div style="width: 60px;"></div>
            </div>
        </div>
    </header>

    <!-- MAIN CONTENT -->
    <main>
        <section class="model-info-section">
            <div class="container">
                <div class="model-info-grid">
                    <!-- LEFT SIDE: Model Details -->
                    <aside class="model-details">
                        <div class="details-header">
                            <h2 class="details-title">Model Details</h2>
                        </div>
                        
                        <!-- Measurements -->
                        <ul class="measurements-list">
                            <?php 
                            $measurements = [
                                'height' => 'Height',
                                'chest' => 'Chest',
                                'waist' => 'Waist', 
                                'hips' => 'Hips',
                                'shoe' => 'Shoe Size',
                                'eyes' => 'Eye Color',
                                'hair' => 'Hair Color'
                            ];
                            
                            foreach ($measurements as $key => $label):
                                $value = getMeasurement($model, $key);
                                if (!empty($value)):
                            ?>
                            <li class="measurement-item">
                                <span class="measurement-label"><?php echo $label; ?></span>
                                <span class="measurement-value"><?php echo htmlspecialchars($value); ?></span>
                            </li>
                            <?php 
                                endif;
                            endforeach; 
                            ?>
                        </ul>
                        
                        <!-- Contact Information -->
                        <div class="contact-info">
                            <?php if (!empty($instagram_url)): ?>
                            <a href="<?php echo htmlspecialchars($instagram_url); ?>" target="_blank" class="contact-item">
                                <div class="contact-icon">
                                    <i class="fab fa-instagram"></i>
                                </div>
                                <div class="contact-text">
                                    <div class="contact-label">Instagram</div>
                                    <div class="contact-value">
                                        <?php echo !empty($instagram_handle) ? htmlspecialchars($instagram_handle) : 'Follow on Instagram'; ?>
                                    </div>
                                </div>
                            </a>
                            <?php endif; ?>
                            
                            <?php if (!empty($contact_info)): ?>
                            <div class="contact-item">
                                <div class="contact-icon">
                                    <i class="fas fa-phone"></i>
                                </div>
                                <div class="contact-text">
                                    <div class="contact-label">Contact</div>
                                    <div class="contact-value"><?php echo htmlspecialchars($contact_info); ?></div>
                                </div>
                            </div>
                            <?php endif; ?>
                            
                            <?php if (!empty($extracted_details)): ?>
                            <div class="contact-item">
                                <div class="contact-icon">
                                    <i class="fas fa-info-circle"></i>
                                </div>
                                <div class="contact-text">
                                    <div class="contact-label">Details</div>
                                    <div class="contact-value">
                                        <?php 
                                        $short_details = strlen($extracted_details) > 100 
                                            ? substr($extracted_details, 0, 100) . '...' 
                                            : $extracted_details;
                                        echo htmlspecialchars($short_details);
                                        ?>
                                    </div>
                                </div>
                            </div>
                            <?php endif; ?>
                        </div>
                    </aside>
                    
                    <!-- RIGHT SIDE: Image Gallery -->
                    <section class="gallery-section">
                        <h2 class="gallery-title">Portfolio Gallery</h2>
                        
                        <div class="image-grid">
                            <?php foreach ($images as $index => $image): ?>
                            <div class="image-item">
                                <img src="<?php echo base_url($image['image_path']); ?>" 
                                     alt="<?php echo htmlspecialchars($image['image_name']); ?>"
                                     loading="lazy">
                                <div class="image-overlay">
                                    <span class="image-number"><?php echo $index + 1; ?>/<?php echo count($images); ?></span>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </section>
                </div>
            </div>
        </section>
    </main>

    <!-- FOOTER -->
    <footer class="portfolio-footer">
        <div class="container">
            <img src="<?php echo base_url('assets/images/logogo.png'); ?>" 
                 alt="GO Productions" 
                 class="footer-logo">
            <p class="footer-text">GO PRODUCTIONS</p>
            <p class="copyright">© <?php echo date('Y'); ?> GO Productions. All rights reserved.</p>
        </div>
    </footer>

    <!-- ACTION BUTTONS -->
    <div class="action-buttons">
       
        <button onclick="scrollToTop()" class="action-btn" title="Back to Top" id="backToTopBtn">
            <i class="fas fa-arrow-up"></i>
        </button>
       
    </div>

    <script>
        // Back to Top Button
        const backToTopBtn = document.getElementById('backToTopBtn');
        
        window.addEventListener('scroll', function() {
            if (window.scrollY > 300) {
                backToTopBtn.style.display = 'flex';
            } else {
                backToTopBtn.style.display = 'flex'; // Always show
            }
        });

        function scrollToTop() {
            window.scrollTo({
                top: 0,
                behavior: 'smooth'
            });
        }

        // Share Portfolio
      

        // Image Lightbox (optional enhancement)
        document.querySelectorAll('.image-item img').forEach(img => {
            img.addEventListener('click', function() {
                const lightbox = document.createElement('div');
                lightbox.style.cssText = `
                    position: fixed;
                    top: 0;
                    left: 0;
                    width: 100%;
                    height: 100%;
                    background: rgba(0,0,0,0.9);
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    z-index: 2000;
                    cursor: pointer;
                `;
                
                const lightboxImg = document.createElement('img');
                lightboxImg.src = this.src;
                lightboxImg.style.cssText = `
                    max-width: 90%;
                    max-height: 90%;
                    object-fit: contain;
                `;
                
                lightbox.appendChild(lightboxImg);
                document.body.appendChild(lightbox);
                
                lightbox.addEventListener('click', function() {
                    document.body.removeChild(lightbox);
                });
            });
        });

        // Initialize
        document.addEventListener('DOMContentLoaded', function() {
            console.log('Portfolio loaded: <?php echo $model_name; ?>');
            console.log('Total images: <?php echo count($images); ?>');
        });
    </script>
</body>
</html>