<main class="main-content" id="mainContent">
    <header class="main-header">
        <h1>SEARCH PROJECTS</h1>
        <p class="welcome-text">Discover and filter projects with advanced search capabilities</p>
    </header>

    <div class="content-area">
        <!-- Modern Search Header -->
        <div class="modern-search-header">
            <div class="search-hero">
                <div class="search-hero-content">
                    <h2>Find Your Projects</h2>
                    <p>Search across projects, models, and extracted data</p>
                    
                    <div class="main-search-container">
                        <div class="search-input-wrapper">
                            <i class="fas fa-search search-hero-icon"></i>
                            <input type="text" id="searchInput" 
                                   placeholder="Search projects, model names, Instagram handles, descriptions..."
                                   class="modern-search-input">
                            <button class="search-clear" onclick="clearSearch()">
                                <i class="fas fa-times"></i>
                            </button>
                        </div>
                        <div id="searchSuggestions" class="modern-search-suggestions"></div>
                    </div>
                </div>
                
                <div class="search-stats-panel">
                    <div class="stat-card">
                        <div class="stat-icon">
                            <i class="fas fa-folder"></i>
                        </div>
                        <div class="stat-content">
                            <div class="stat-value" id="totalProjects">0</div>
                            <div class="stat-label">Total Projects</div>
                        </div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-icon">
                            <i class="fas fa-users"></i>
                        </div>
                        <div class="stat-content">
                            <div class="stat-value" id="totalModels">0</div>
                            <div class="stat-label">Models Found</div>
                        </div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-icon">
                            <i class="fas fa-images"></i>
                        </div>
                        <div class="stat-content">
                            <div class="stat-value" id="totalImages">0</div>
                            <div class="stat-label">Total Images</div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Modern Filters Section -->
        <div class="modern-filters-section">
            <div class="filters-header">
                <div class="filters-title">
                    <i class="fas fa-sliders-h"></i>
                    <h3>Filters & Options</h3>
                </div>
                <div class="filters-controls">
                    <button onclick="toggleFilters()" class="btn btn-outline filter-toggle">
                        <i class="fas fa-filter"></i>
                        <span>Toggle Filters</span>
                    </button>
                    <button onclick="clearAllFilters()" class="btn btn-outline danger">
                        <i class="fas fa-eraser"></i>
                        <span>Clear All</span>
                    </button>
                </div>
            </div>

            <div id="filtersContent" class="filters-content">
                <div class="filters-grid">
                    <!-- Status Filter -->
                    <div class="filter-group modern">
                        <label class="filter-label">
                            <i class="fas fa-tasks"></i>
                            Project Status
                        </label>
                        <div class="filter-options">
                            <label class="filter-option">
                                <input type="checkbox" value="completed" onchange="toggleStatusFilter(this)">
                                <span class="checkmark"></span>
                                <span class="option-label">Completed</span>
                                <span class="status-indicator completed"></span>
                            </label>
                            <label class="filter-option">
                                <input type="checkbox" value="processing" onchange="toggleStatusFilter(this)">
                                <span class="checkmark"></span>
                                <span class="option-label">Processing</span>
                                <span class="status-indicator processing"></span>
                            </label>
                            <label class="filter-option">
                                <input type="checkbox" value="failed" onchange="toggleStatusFilter(this)">
                                <span class="checkmark"></span>
                                <span class="option-label">Failed</span>
                                <span class="status-indicator failed"></span>
                            </label>
                        </div>
                    </div>

                    <!-- Date Range Filter -->
                    <div class="filter-group modern">
                        <label class="filter-label">
                            <i class="fas fa-calendar"></i>
                            Date Created
                        </label>
                        <div class="date-range-picker">
                            <div class="date-input-group">
                                <label>From</label>
                                <input type="date" id="dateFrom" onchange="applyFilters()" class="date-input">
                            </div>
                            <div class="date-input-group">
                                <label>To</label>
                                <input type="date" id="dateTo" onchange="applyFilters()" class="date-input">
                            </div>
                        </div>
                        <div class="quick-dates">
                            <button class="quick-date-btn" onclick="setQuickDate('today')">Today</button>
                            <button class="quick-date-btn" onclick="setQuickDate('week')">This Week</button>
                            <button class="quick-date-btn" onclick="setQuickDate('month')">This Month</button>
                        </div>
                    </div>

                    <!-- Model Count Filter -->
                    <div class="filter-group modern">
                        <label class="filter-label">
                            <i class="fas fa-user-friends"></i>
                            Models Count
                        </label>
                        <div class="range-slider-container">
                            <div class="range-values">
                                <span id="modelMinValue">0</span>
                                <span>to</span>
                                <span id="modelMaxValue">50+</span>
                            </div>
                            <div class="range-slider">
                                <input type="range" id="modelCountMin" min="0" max="50" value="0" onchange="updateModelRange()">
                                <input type="range" id="modelCountMax" min="0" max="50" value="50" onchange="updateModelRange()">
                            </div>
                        </div>
                    </div>

                    <!-- Image Count Filter -->
                    <div class="filter-group modern">
                        <label class="filter-label">
                            <i class="fas fa-image"></i>
                            Minimum Images
                        </label>
                        <div class="image-count-selector">
                            <div class="count-options">
                                <label class="count-option">
                                    <input type="radio" name="imageCount" value="0" checked onchange="applyFilters()">
                                    <span class="count-bubble">Any</span>
                                </label>
                                <label class="count-option">
                                    <input type="radio" name="imageCount" value="1" onchange="applyFilters()">
                                    <span class="count-bubble">1+</span>
                                </label>
                                <label class="count-option">
                                    <input type="radio" name="imageCount" value="5" onchange="applyFilters()">
                                    <span class="count-bubble">5+</span>
                                </label>
                                <label class="count-option">
                                    <input type="radio" name="imageCount" value="10" onchange="applyFilters()">
                                    <span class="count-bubble">10+</span>
                                </label>
                                <label class="count-option">
                                    <input type="radio" name="imageCount" value="20" onchange="applyFilters()">
                                    <span class="count-bubble">20+</span>
                                </label>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Advanced Filters -->
                <div class="advanced-filters">
                    <div class="advanced-filters-header">
                        <h4>Advanced Filters</h4>
                        <button class="btn-text" onclick="toggleAdvancedFilters()">
                            <i class="fas fa-chevron-down"></i>
                            <span>Show More</span>
                        </button>
                    </div>
                    <div id="advancedFiltersContent" class="advanced-filters-content">
                        <div class="advanced-filters-grid">
                            <div class="filter-group">
                                <label class="filter-label">Sort By</label>
                                <select id="sortBy" onchange="applyFilters()" class="modern-select">
                                    <option value="newest">Newest First</option>
                                    <option value="oldest">Oldest First</option>
                                    <option value="name_asc">Name A-Z</option>
                                    <option value="name_desc">Name Z-A</option>
                                    <option value="models_asc">Fewest Models</option>
                                    <option value="models_desc">Most Models</option>
                                    <option value="images_asc">Fewest Images</option>
                                    <option value="images_desc">Most Images</option>
                                </select>
                            </div>
                            <div class="filter-group">
                                <label class="filter-label">Results Per Page</label>
                                <select id="pageSize" onchange="applyFilters()" class="modern-select">
                                    <option value="12">12 items</option>
                                    <option value="24">24 items</option>
                                    <option value="48">48 items</option>
                                    <option value="96">96 items</option>
                                </select>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Active Filters -->
            <div id="activeFilters" class="modern-active-filters">
                <div class="active-filters-header">
                    <span class="active-filters-title">Active Filters</span>
                    <span id="activeFiltersCount" class="filters-count">0</span>
                </div>
                <div id="filterTags" class="modern-filter-tags"></div>
            </div>
        </div>

        <!-- Results Section -->
        <div class="modern-results-section">
            <div class="results-header-modern">
                <div class="results-info">
                    <h3>Search Results</h3>
                    <div class="results-meta">
                        <span id="resultsCount" class="results-count">0 projects found</span>
                        <span class="results-time" id="searchTime"></span>
                    </div>
                </div>
                <div class="results-actions">
                    <button class="btn btn-outline" onclick="exportResults()">
                        <i class="fas fa-download"></i>
                        Export Results
                    </button>
                    <div class="view-toggle">
                        <button class="view-btn active" data-view="grid" onclick="changeView('grid')">
                            <i class="fas fa-th"></i>
                        </button>
                        <button class="view-btn" data-view="list" onclick="changeView('list')">
                            <i class="fas fa-list"></i>
                        </button>
                    </div>
                </div>
            </div>

            <!-- Loading State -->
            <div id="loadingResults" class="modern-loading">
                <div class="loading-spinner">
                    <div class="spinner-ring"></div>
                </div>
                <div class="loading-content">
                    <h4>Searching Projects</h4>
                    <p>We're finding the best matches for your search...</p>
                    <div class="loading-progress">
                        <div class="progress-bar">
                            <div class="progress-fill" id="loadingProgress"></div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Results Grid/List -->
            <div id="resultsContainer" class="results-container grid-view">
                <div id="resultsGrid" class="modern-results-grid">
                    <!-- Results will be populated here by JavaScript -->
                </div>
            </div>

            <!-- No Results State -->
            <div id="noResults" class="modern-no-results">
                <div class="no-results-illustration">
                    <i class="fas fa-search-minus"></i>
                </div>
                <div class="no-results-content">
                    <h3>No projects found</h3>
                    <p>We couldn't find any projects matching your search criteria. Try adjusting your filters or search terms.</p>
                    <div class="no-results-actions">
                        <button onclick="clearAllFilters()" class="btn btn-primary">
                            <i class="fas fa-eraser"></i>
                            Clear All Filters
                        </button>
                        <button onclick="suggestSimilar()" class="btn btn-outline">
                            <i class="fas fa-lightbulb"></i>
                            Show Similar Projects
                        </button>
                    </div>
                </div>
            </div>

            <!-- Pagination -->
            <div id="pagination" class="modern-pagination">
                <!-- Pagination will be populated here by JavaScript -->
            </div>
        </div>
    </div>
</main>
<style> 
/* Modern Search Header */
.modern-search-header {
    background: var(--card-bg);
    border-radius: 16px;
    margin-bottom: 30px;
    border: 1px solid var(--border-color);
    overflow: hidden;
}

.search-hero {
    padding: 40px;
    background: linear-gradient(135deg, var(--card-bg) 0%, rgba(0,0,0,0.02) 100%);
}

.search-hero-content {
    max-width: 800px;
    margin: 0 auto;
    text-align: center;
}

.search-hero-content h2 {
    font-size: 2.5rem;
    font-weight: 800;
    color: var(--text-color);
    margin-bottom: 10px;
    background: linear-gradient(135deg, var(--text-color) 0%, var(--light-text) 100%);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
}

.search-hero-content p {
    font-size: 1.1rem;
    color: var(--light-text);
    margin-bottom: 30px;
}

.main-search-container {
    position: relative;
    max-width: 600px;
    margin: 0 auto;
}

.search-input-wrapper {
    position: relative;
    display: flex;
    align-items: center;
    background: var(--card-bg);
    border: 2px solid var(--border-color);
    border-radius: 12px;
    padding: 0 15px;
    transition: all 0.3s ease;
}

.search-input-wrapper:focus-within {
    border-color: var(--black);
    box-shadow: 0 8px 25px rgba(0,0,0,0.1);
}

.search-hero-icon {
    font-size: 18px;
    color: var(--light-text);
    margin-right: 10px;
}

.modern-search-input {
    flex: 1;
    padding: 15px 0;
    border: none;
    background: none;
    font-size: 16px;
    color: var(--text-color);
    outline: none;
}

.modern-search-input::placeholder {
    color: var(--light-text);
}

.search-clear {
    display: none;
    background: none;
    border: none;
    color: var(--light-text);
    cursor: pointer;
    padding: 5px;
    border-radius: 50%;
    transition: all 0.3s ease;
}

.search-clear:hover {
    background: var(--border-color);
    color: var(--text-color);
}

/* Search Stats Panel */
.search-stats-panel {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 20px;
    margin-top: 40px;
    padding-top: 30px;
    border-top: 1px solid var(--border-color);
}

.stat-card {
    display: flex;
    align-items: center;
    padding: 20px;
    background: var(--card-bg);
    border: 1px solid var(--border-color);
    border-radius: 12px;
    transition: all 0.3s ease;
}

.stat-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(0,0,0,0.1);
}

.stat-icon {
    width: 50px;
    height: 50px;
    background: var(--black);
    color: white;
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 20px;
    margin-right: 15px;
}

.stat-content {
    flex: 1;
}

.stat-value {
    font-size: 1.8rem;
    font-weight: 800;
    color: var(--text-color);
    line-height: 1;
}

.stat-label {
    font-size: 0.9rem;
    color: var(--light-text);
    margin-top: 5px;
}

/* Modern Filters Section */
.modern-filters-section {
    background: var(--card-bg);
    border-radius: 16px;
    margin-bottom: 30px;
    border: 1px solid var(--border-color);
    overflow: hidden;
}

.filters-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 25px 30px;
    border-bottom: 1px solid var(--border-color);
}

.filters-title {
    display: flex;
    align-items: center;
    gap: 12px;
}

.filters-title i {
    font-size: 20px;
    color: var(--light-text);
}

.filters-title h3 {
    margin: 0;
    color: var(--text-color);
    font-weight: 700;
}

.filters-controls {
    display: flex;
    gap: 10px;
}

.filter-toggle {
    display: none;
}

.filters-content {
    padding: 30px;
}

.filters-content.collapsed {
    display: none;
}

.filters-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 30px;
    margin-bottom: 30px;
}

.filter-group.modern {
    background: var(--card-bg);
    border: 1px solid var(--border-color);
    border-radius: 12px;
    padding: 20px;
}

.filter-label {
    display: flex;
    align-items: center;
    gap: 8px;
    font-weight: 600;
    color: var(--text-color);
    margin-bottom: 15px;
    font-size: 0.95rem;
}

.filter-label i {
    color: var(--light-text);
    width: 16px;
}

/* Status Filter Options */
.filter-options {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.filter-option {
    display: flex;
    align-items: center;
    gap: 10px;
    padding: 8px 0;
    cursor: pointer;
    transition: all 0.3s ease;
}

.filter-option:hover {
    background: var(--hover-color);
    margin: 0 -10px;
    padding: 8px 10px;
    border-radius: 6px;
}

.filter-option input {
    display: none;
}

.checkmark {
    width: 18px;
    height: 18px;
    border: 2px solid var(--border-color);
    border-radius: 4px;
    position: relative;
    transition: all 0.3s ease;
}

.filter-option input:checked + .checkmark {
    background: var(--black);
    border-color: var(--black);
}

.filter-option input:checked + .checkmark::after {
    content: '✓';
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    color: white;
    font-size: 12px;
    font-weight: bold;
}

.option-label {
    flex: 1;
    color: var(--text-color);
}

.status-indicator {
    width: 8px;
    height: 8px;
    border-radius: 50%;
}

.status-indicator.completed { background: #28a745; }
.status-indicator.processing { background: #ffc107; }
.status-indicator.failed { background: #dc3545; }

/* Date Range Picker */
.date-range-picker {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 15px;
    margin-bottom: 15px;
}

.date-input-group {
    display: flex;
    flex-direction: column;
    gap: 5px;
}

.date-input-group label {
    font-size: 0.8rem;
    color: var(--light-text);
    font-weight: 500;
}

.date-input {
    padding: 10px;
    border: 1px solid var(--border-color);
    border-radius: 6px;
    background: var(--card-bg);
    color: var(--text-color);
    font-size: 14px;
}

.quick-dates {
    display: flex;
    gap: 8px;
}

.quick-date-btn {
    padding: 6px 12px;
    border: 1px solid var(--border-color);
    background: var(--card-bg);
    color: var(--text-color);
    border-radius: 6px;
    font-size: 0.8rem;
    cursor: pointer;
    transition: all 0.3s ease;
}

.quick-date-btn:hover {
    background: var(--black);
    color: white;
    border-color: var(--black);
}

/* Range Slider */
.range-slider-container {
    margin-top: 10px;
}

.range-values {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 10px;
    font-size: 0.9rem;
    color: var(--text-color);
}

.range-slider {
    position: relative;
    height: 4px;
    background: var(--border-color);
    border-radius: 2px;
}

.range-slider input[type="range"] {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    pointer-events: none;
    opacity: 0;
}

.range-slider::before {
    content: '';
    position: absolute;
    top: 0;
    left: var(--min-percent, 0%);
    right: calc(100% - var(--max-percent, 100%));
    height: 100%;
    background: var(--black);
    border-radius: 2px;
}

/* Image Count Selector */
.count-options {
    display: flex;
    gap: 8px;
    flex-wrap: wrap;
}

.count-option {
    cursor: pointer;
}

.count-option input {
    display: none;
}

.count-bubble {
    display: block;
    padding: 8px 16px;
    border: 2px solid var(--border-color);
    border-radius: 20px;
    background: var(--card-bg);
    color: var(--text-color);
    font-size: 0.9rem;
    font-weight: 500;
    transition: all 0.3s ease;
    text-align: center;
    min-width: 50px;
}

.count-option input:checked + .count-bubble {
    background: var(--black);
    color: white;
    border-color: var(--black);
}

/* Advanced Filters */
.advanced-filters {
    border-top: 1px solid var(--border-color);
    padding-top: 25px;
}

.advanced-filters-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 20px;
}

.advanced-filters-header h4 {
    margin: 0;
    color: var(--text-color);
}

.advanced-filters-content {
    display: block;
}

.advanced-filters-content.collapsed {
    display: none;
}

.advanced-filters-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 20px;
}

.modern-select {
    width: 100%;
    padding: 10px 12px;
    border: 1px solid var(--border-color);
    border-radius: 6px;
    background: var(--card-bg);
    color: var(--text-color);
    font-size: 14px;
}

/* Modern Active Filters */
.modern-active-filters {
    display: none;
    background: var(--hover-color);
    padding: 20px 30px;
    border-top: 1px solid var(--border-color);
}

.active-filters-header {
    display: flex;
    align-items: center;
    gap: 10px;
    margin-bottom: 15px;
}

.active-filters-title {
    font-weight: 600;
    color: var(--text-color);
}

.filters-count {
    background: var(--black);
    color: white;
    padding: 2px 8px;
    border-radius: 12px;
    font-size: 0.8rem;
    font-weight: 600;
}

.modern-filter-tags {
    display: flex;
    flex-wrap: wrap;
    gap: 8px;
}

.filter-tag-modern {
    display: flex;
    align-items: center;
    gap: 8px;
    background: var(--card-bg);
    border: 1px solid var(--border-color);
    border-radius: 20px;
    padding: 6px 12px;
    font-size: 0.85rem;
}

.filter-tag-label {
    color: var(--text-color);
}

.filter-tag-remove {
    background: none;
    border: none;
    color: var(--light-text);
    cursor: pointer;
    padding: 2px;
    border-radius: 50%;
    transition: all 0.3s ease;
}

.filter-tag-remove:hover {
    background: var(--border-color);
    color: var(--text-color);
}

/* Modern Results Section */
.modern-results-section {
    background: var(--card-bg);
    border-radius: 16px;
    border: 1px solid var(--border-color);
    overflow: hidden;
}

.results-header-modern {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    padding: 25px 30px;
    border-bottom: 1px solid var(--border-color);
}

.results-info h3 {
    margin: 0 0 5px 0;
    color: var(--text-color);
    font-weight: 700;
}

.results-meta {
    display: flex;
    gap: 15px;
    align-items: center;
}

.results-count {
    color: var(--text-color);
    font-weight: 500;
}

.results-time {
    color: var(--light-text);
    font-size: 0.9rem;
}

.results-actions {
    display: flex;
    gap: 15px;
    align-items: center;
}

.view-toggle {
    display: flex;
    background: var(--hover-color);
    border-radius: 8px;
    padding: 4px;
}

.view-btn {
    background: none;
    border: none;
    padding: 8px 12px;
    color: var(--light-text);
    cursor: pointer;
    border-radius: 6px;
    transition: all 0.3s ease;
}

.view-btn.active {
    background: var(--card-bg);
    color: var(--text-color);
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
}

/* Results Container */
.results-container {
    padding: 30px;
}

.modern-results-grid {
    display: grid;
    gap: 25px;
}

.grid-view .modern-results-grid {
    grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
}

.list-view .modern-results-grid {
    grid-template-columns: 1fr;
}

.list-view .project-card-modern {
    display: flex;
    align-items: center;
    padding: 20px;
}

.list-view .project-card-content {
    flex: 1;
    margin-right: 20px;
}

.list-view .project-stats-modern {
    justify-content: flex-start;
    gap: 25px;
}

/* Modern Project Cards */
.project-card-modern {
    background: var(--card-bg);
    border: 1px solid var(--border-color);
    border-radius: 12px;
    overflow: hidden;
    transition: all 0.3s ease;
    position: relative;
}

.project-card-modern:hover {
    transform: translateY(-4px);
    box-shadow: 0 12px 40px rgba(0,0,0,0.15);
    border-color: var(--black);
}

.project-card-header {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    padding: 20px 20px 0 20px;
}

.project-badge {
    font-size: 0.75rem;
    font-weight: 600;
    padding: 4px 8px;
    border-radius: 6px;
    background: transparent;
    color: var(--light-text);
}

.project-badge.new {
    background: #ff6b6b;
    color: white;
}

.project-badge.new i {
    margin-right: 4px;
}

.project-actions {
    display: flex;
    gap: 5px;
}

.btn-icon.small {
    width: 28px;
    height: 28px;
    font-size: 12px;
}

.project-card-content {
    padding: 20px;
}

.project-title {
    font-size: 1.1rem;
    font-weight: 700;
    color: var(--text-color);
    margin: 0 0 10px 0;
    line-height: 1.3;
}

.project-description {
    color: var(--light-text);
    font-size: 0.9rem;
    line-height: 1.5;
    margin-bottom: 20px;
    display: -webkit-box;
    -webkit-line-clamp: 2;
    -webkit-box-orient: vertical;
    overflow: hidden;
}

.project-stats-modern {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 20px;
}

.stat-item-modern {
    display: flex;
    flex-direction: column;
    align-items: center;
    text-align: center;
    gap: 5px;
}

.stat-item-modern i {
    color: var(--light-text);
    font-size: 1rem;
}

.stat-value {
    font-size: 1.1rem;
    font-weight: 700;
    color: var(--text-color);
}

.stat-label {
    font-size: 0.75rem;
    color: var(--light-text);
}

.project-models-preview {
    margin-bottom: 20px;
}

.models-scroll {
    display: flex;
    gap: 8px;
    overflow-x: auto;
    padding-bottom: 5px;
}

.models-scroll::-webkit-scrollbar {
    height: 3px;
}

.models-scroll::-webkit-scrollbar-track {
    background: var(--border-color);
    border-radius: 2px;
}

.models-scroll::-webkit-scrollbar-thumb {
    background: var(--light-text);
    border-radius: 2px;
}

.model-chip {
    display: flex;
    flex-direction: column;
    background: var(--hover-color);
    padding: 6px 10px;
    border-radius: 8px;
    font-size: 0.8rem;
    white-space: nowrap;
}

.model-name {
    font-weight: 600;
    color: var(--text-color);
}

.model-handle {
    color: var(--light-text);
    font-size: 0.7rem;
}

.more-models {
    background: var(--border-color);
    padding: 6px 10px;
    border-radius: 8px;
    font-size: 0.8rem;
    color: var(--light-text);
    white-space: nowrap;
}

.project-card-footer {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 15px 20px;
    background: var(--hover-color);
    border-top: 1px solid var(--border-color);
}

.project-status {
    display: flex;
    align-items: center;
    gap: 6px;
    font-size: 0.8rem;
    font-weight: 600;
    padding: 4px 10px;
    border-radius: 12px;
    background: var(--card-bg);
}

.project-status i {
    font-size: 0.6rem;
}

.status-completed { color: #28a745; }
.status-processing { color: #ffc107; }
.status-failed { color: #dc3545; }

.btn.small {
    padding: 8px 16px;
    font-size: 0.85rem;
}

/* Modern Loading State */
.modern-loading {
    display: none;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    padding: 60px 30px;
    text-align: center;
}

.loading-spinner {
    margin-bottom: 25px;
}

.spinner-ring {
    width: 60px;
    height: 60px;
    border: 3px solid var(--border-color);
    border-top: 3px solid var(--black);
    border-radius: 50%;
    animation: spin 1s linear infinite;
}

.loading-content h4 {
    margin: 0 0 10px 0;
    color: var(--text-color);
    font-weight: 600;
}

.loading-content p {
    color: var(--light-text);
    margin-bottom: 20px;
}

.loading-progress {
    width: 200px;
}

.loading-progress .progress-bar {
    height: 4px;
    background: var(--border-color);
    border-radius: 2px;
    overflow: hidden;
}

.loading-progress .progress-fill {
    height: 100%;
    background: var(--black);
    transition: width 0.3s ease;
}

/* Modern No Results */
.modern-no-results {
    display: none;
    text-align: center;
    padding: 80px 30px;
}

.no-results-illustration {
    font-size: 80px;
    color: var(--border-color);
    margin-bottom: 30px;
}

.no-results-content h3 {
    margin: 0 0 15px 0;
    color: var(--text-color);
    font-weight: 600;
}

.no-results-content p {
    color: var(--light-text);
    margin-bottom: 30px;
    max-width: 400px;
    margin-left: auto;
    margin-right: auto;
}

.no-results-actions {
    display: flex;
    gap: 15px;
    justify-content: center;
    flex-wrap: wrap;
}

/* Modern Pagination */
.modern-pagination {
    display: none;
    justify-content: center;
    align-items: center;
    gap: 8px;
    padding: 30px;
    border-top: 1px solid var(--border-color);
}

.pagination-btn {
    display: flex;
    align-items: center;
    justify-content: center;
    width: 40px;
    height: 40px;
    border: 1px solid var(--border-color);
    background: var(--card-bg);
    color: var(--text-color);
    border-radius: 8px;
    cursor: pointer;
    transition: all 0.3s ease;
    font-weight: 500;
}

.pagination-btn:hover:not(.disabled):not(.active) {
    border-color: var(--black);
    color: var(--black);
}

.pagination-btn.active {
    background: var(--black);
    color: white;
    border-color: var(--black);
}

.pagination-btn.disabled {
    opacity: 0.5;
    cursor: not-allowed;
}

/* Modern Search Suggestions */
.modern-search-suggestions {
    position: absolute;
    top: 100%;
    left: 0;
    right: 0;
    background: var(--card-bg);
    border: 1px solid var(--border-color);
    border-radius: 12px;
    box-shadow: 0 20px 60px rgba(0,0,0,0.2);
    z-index: 1000;
    max-height: 400px;
    overflow-y: auto;
    display: none;
    margin-top: 5px;
}

.suggestion-item {
    padding: 15px 20px;
    cursor: pointer;
    border-bottom: 1px solid var(--border-color);
    transition: all 0.3s ease;
}

.suggestion-item:hover {
    background: var(--hover-color);
}

.suggestion-item:last-child {
    border-bottom: none;
}

.suggestion-main {
    display: flex;
    align-items: flex-start;
    gap: 12px;
    margin-bottom: 8px;
}

.suggestion-main i {
    color: var(--light-text);
    margin-top: 2px;
    flex-shrink: 0;
}

.suggestion-content {
    flex: 1;
}

.suggestion-title {
    font-weight: 600;
    color: var(--text-color);
    margin-bottom: 4px;
}

.suggestion-meta {
    font-size: 0.85rem;
    color: var(--light-text);
}

.suggestion-models {
    font-size: 0.8rem;
    color: var(--light-text);
    padding-left: 28px;
}

/* Notifications */
.notification {
    position: fixed;
    top: 20px;
    right: 20px;
    background: var(--card-bg);
    border: 1px solid var(--border-color);
    border-radius: 8px;
    padding: 15px 20px;
    box-shadow: 0 10px 30px rgba(0,0,0,0.2);
    z-index: 10000;
    animation: slideInRight 0.3s ease;
}

.notification-content {
    display: flex;
    align-items: center;
    gap: 10px;
}

.notification-success i { color: #28a745; }
.notification-info i { color: #17a2b8; }

@keyframes slideInRight {
    from {
        transform: translateX(100%);
        opacity: 0;
    }
    to {
        transform: translateX(0);
        opacity: 1;
    }
}

/* Responsive Design */
@media (max-width: 768px) {
    .search-hero {
        padding: 30px 20px;
    }
    
    .search-hero-content h2 {
        font-size: 2rem;
    }
    
    .search-stats-panel {
        grid-template-columns: 1fr;
        gap: 15px;
    }
    
    .filters-header {
        flex-direction: column;
        gap: 15px;
        align-items: stretch;
    }
    
    .filter-toggle {
        display: flex;
    }
    
    .filters-content {
        padding: 20px;
    }
    
    .filters-grid {
        grid-template-columns: 1fr;
        gap: 20px;
    }
    
    .date-range-picker {
        grid-template-columns: 1fr;
    }
    
    .results-header-modern {
        flex-direction: column;
        gap: 15px;
    }
    
    .results-actions {
        width: 100%;
        justify-content: space-between;
    }
    
    .grid-view .modern-results-grid {
        grid-template-columns: 1fr;
    }
    
    .list-view .project-card-modern {
        flex-direction: column;
        align-items: stretch;
    }
    
    .list-view .project-card-content {
        margin-right: 0;
        margin-bottom: 15px;
    }
    
    .no-results-actions {
        flex-direction: column;
        align-items: center;
    }
    
    .modern-active-filters {
        padding: 15px 20px;
    }
}

@media (max-width: 480px) {
    .search-hero {
        padding: 20px 15px;
    }
    
    .search-hero-content h2 {
        font-size: 1.5rem;
    }
    
    .stat-card {
        padding: 15px;
    }
    
    .stat-icon {
        width: 40px;
        height: 40px;
        font-size: 16px;
    }
    
    .stat-value {
        font-size: 1.5rem;
    }
    
    .filters-content {
        padding: 15px;
    }
    
    .results-container {
        padding: 20px 15px;
    }
    
    .project-card-content {
        padding: 15px;
    }
    
    .project-stats-modern {
        flex-direction: column;
        gap: 15px;
        align-items: stretch;
    }
    
    .stat-item-modern {
        flex-direction: row;
        justify-content: space-between;
        text-align: left;
    }
}
</style>

<script>
// Global variables
let allProjects = [];
let searchTimeout;
let currentView = 'grid';
let currentPage = 1;
let pageSize = 12;
let currentFilters = {
    search: '',
    status: [],
    dateFrom: '',
    dateTo: '',
    modelMin: 0,
    modelMax: 50,
    imageCount: '0',
    sort: 'newest'
};

// Initialize search functionality
document.addEventListener('DOMContentLoaded', function() {
    loadAllProjects();
    setupSearch();
    initializeFilters();
});

function initializeFilters() {
    // Set default dates
    const today = new Date().toISOString().split('T')[0];
    const monthAgo = new Date();
    monthAgo.setMonth(monthAgo.getMonth() - 1);
    document.getElementById('dateTo').value = today;
    document.getElementById('dateFrom').value = monthAgo.toISOString().split('T')[0];
    
    updateModelRange();
}

function loadAllProjects() {
    // Show loading state
    showLoading();
    
    // Simulate API call - replace with actual API endpoint
    fetch('<?php echo site_url("project/get_all_projects_data"); ?>')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                allProjects = data.projects;
                updateQuickStats();
                applyFilters();
            }
        })
        .catch(error => {
            console.error('Error loading projects:', error);
            hideLoading();
        });
}

function setupSearch() {
    const searchInput = document.getElementById('searchInput');
    
    searchInput.addEventListener('input', function(e) {
        const searchTerm = e.target.value.trim();
        currentFilters.search = searchTerm;
        
        // Show/hide clear button
        const clearBtn = document.querySelector('.search-clear');
        clearBtn.style.display = searchTerm ? 'flex' : 'none';
        
        // Clear previous timeout
        clearTimeout(searchTimeout);
        
        // Show suggestions for non-empty search
        if (searchTerm.length > 1) {
            showSearchSuggestions(searchTerm);
        } else {
            hideSearchSuggestions();
        }
        
        // Debounce search execution
        searchTimeout = setTimeout(() => {
            applyFilters();
        }, 500);
    });
    
    // Hide suggestions when clicking outside
    document.addEventListener('click', function(e) {
        if (!e.target.closest('.main-search-container')) {
            hideSearchSuggestions();
        }
    });
}

function clearSearch() {
    document.getElementById('searchInput').value = '';
    currentFilters.search = '';
    document.querySelector('.search-clear').style.display = 'none';
    hideSearchSuggestions();
    applyFilters();
}

function showSearchSuggestions(searchTerm) {
    const suggestionsContainer = document.getElementById('searchSuggestions');
    const matchingProjects = allProjects.filter(project => 
        project.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
        project.models.some(model => 
            model.model_name.toLowerCase().includes(searchTerm.toLowerCase()) ||
            (model.instagram_handle && model.instagram_handle.toLowerCase().includes(searchTerm.toLowerCase()))
        )
    ).slice(0, 8);
    
    if (matchingProjects.length === 0) {
        hideSearchSuggestions();
        return;
    }
    
    let suggestionsHTML = '';
    matchingProjects.forEach(project => {
        const matchingModels = project.models.filter(model =>
            model.model_name.toLowerCase().includes(searchTerm.toLowerCase()) ||
            (model.instagram_handle && model.instagram_handle.toLowerCase().includes(searchTerm.toLowerCase()))
        );
        
        suggestionsHTML += `
            <div class="suggestion-item" onclick="selectSuggestion('${project.name}')">
                <div class="suggestion-main">
                    <i class="fas fa-folder"></i>
                    <div class="suggestion-content">
                        <div class="suggestion-title">${highlightMatch(project.name, searchTerm)}</div>
                        <div class="suggestion-meta">
                            ${project.models.length} models • ${getTotalImages(project)} images
                        </div>
                    </div>
                </div>
                ${matchingModels.length > 0 ? `
                    <div class="suggestion-models">
                        <strong>Matching models:</strong> 
                        ${matchingModels.slice(0, 2).map(m => m.model_name).join(', ')}
                        ${matchingModels.length > 2 ? ` +${matchingModels.length - 2} more` : ''}
                    </div>
                ` : ''}
            </div>
        `;
    });
    
    suggestionsContainer.innerHTML = suggestionsHTML;
    suggestionsContainer.style.display = 'block';
}

function hideSearchSuggestions() {
    document.getElementById('searchSuggestions').style.display = 'none';
}

function selectSuggestion(value) {
    document.getElementById('searchInput').value = value;
    currentFilters.search = value;
    hideSearchSuggestions();
    applyFilters();
}

function highlightMatch(text, searchTerm) {
    if (!searchTerm) return text;
    const regex = new RegExp(`(${searchTerm})`, 'gi');
    return text.replace(regex, '<mark>$1</mark>');
}

function getTotalImages(project) {
    return project.models.reduce((sum, model) => sum + (model.image_count || 0), 0);
}

// Filter functions
function toggleStatusFilter(checkbox) {
    const value = checkbox.value;
    if (checkbox.checked) {
        if (!currentFilters.status.includes(value)) {
            currentFilters.status.push(value);
        }
    } else {
        currentFilters.status = currentFilters.status.filter(s => s !== value);
    }
    applyFilters();
}

function updateModelRange() {
    const min = document.getElementById('modelCountMin').value;
    const max = document.getElementById('modelCountMax').value;
    
    document.getElementById('modelMinValue').textContent = min;
    document.getElementById('modelMaxValue').textContent = max === '50' ? '50+' : max;
    
    currentFilters.modelMin = parseInt(min);
    currentFilters.modelMax = parseInt(max);
    applyFilters();
}

function setQuickDate(range) {
    const today = new Date();
    const from = new Date();
    
    switch (range) {
        case 'today':
            from.setDate(today.getDate());
            break;
        case 'week':
            from.setDate(today.getDate() - 7);
            break;
        case 'month':
            from.setMonth(today.getMonth() - 1);
            break;
    }
    
    document.getElementById('dateFrom').value = from.toISOString().split('T')[0];
    document.getElementById('dateTo').value = today.toISOString().split('T')[0];
    
    currentFilters.dateFrom = document.getElementById('dateFrom').value;
    currentFilters.dateTo = document.getElementById('dateTo').value;
    applyFilters();
}

function toggleFilters() {
    const content = document.getElementById('filtersContent');
    content.classList.toggle('collapsed');
}

function toggleAdvancedFilters() {
    const content = document.getElementById('advancedFiltersContent');
    const button = document.querySelector('.advanced-filters-header .btn-text');
    
    content.classList.toggle('collapsed');
    button.innerHTML = content.classList.contains('collapsed') ? 
        '<i class="fas fa-chevron-down"></i><span>Show More</span>' :
        '<i class="fas fa-chevron-up"></i><span>Show Less</span>';
}

// Apply all filters and search
function applyFilters() {
    const startTime = performance.now();
    showLoading();
    
    // Update current filters from UI
    currentFilters.dateFrom = document.getElementById('dateFrom').value;
    currentFilters.dateTo = document.getElementById('dateTo').value;
    currentFilters.imageCount = document.querySelector('input[name="imageCount"]:checked').value;
    currentFilters.sort = document.getElementById('sortBy').value;
    pageSize = parseInt(document.getElementById('pageSize').value);
    
    // Filter projects
    let filteredProjects = allProjects.filter(project => {
        // Search filter
        if (currentFilters.search && !matchesSearch(project, currentFilters.search)) {
            return false;
        }
        
        // Status filter
        if (currentFilters.status.length > 0 && !currentFilters.status.includes(project.status)) {
            return false;
        }
        
        // Date filter
        if (!matchesDate(project.created_at, currentFilters.dateFrom, currentFilters.dateTo)) {
            return false;
        }
        
        // Model count filter
        if (project.models.length < currentFilters.modelMin || 
            (currentFilters.modelMax < 50 && project.models.length > currentFilters.modelMax)) {
            return false;
        }
        
        // Image count filter
        const totalImages = getTotalImages(project);
        if (totalImages < parseInt(currentFilters.imageCount)) {
            return false;
        }
        
        return true;
    });
    
    // Sort projects
    filteredProjects = sortProjects(filteredProjects, currentFilters.sort);
    
    // Calculate pagination
    const totalPages = Math.ceil(filteredProjects.length / pageSize);
    const startIndex = (currentPage - 1) * pageSize;
    const paginatedProjects = filteredProjects.slice(startIndex, startIndex + pageSize);
    
    // Display results
    displayResults(paginatedProjects, filteredProjects.length);
    updateActiveFilters();
    updatePagination(totalPages);
    
    // Update search time
    const endTime = performance.now();
    document.getElementById('searchTime').textContent = `in ${(endTime - startTime).toFixed(0)}ms`;
    
    hideLoading();
}

function matchesSearch(project, searchTerm) {
    const term = searchTerm.toLowerCase();
    return (
        project.name.toLowerCase().includes(term) ||
        project.description?.toLowerCase().includes(term) ||
        project.models.some(model =>
            model.model_name.toLowerCase().includes(term) ||
            (model.instagram_handle && model.instagram_handle.toLowerCase().includes(term)) ||
            (model.extracted_details && model.extracted_details.toLowerCase().includes(term))
        )
    );
}

function matchesDate(createdAt, dateFrom, dateTo) {
    if (!dateFrom && !dateTo) return true;
    
    const created = new Date(createdAt).getTime();
    const from = dateFrom ? new Date(dateFrom).getTime() : 0;
    const to = dateTo ? new Date(dateTo).getTime() + 86400000 : Date.now(); // +1 day to include the end date
    
    return created >= from && created <= to;
}

function sortProjects(projects, sortBy) {
    return [...projects].sort((a, b) => {
        switch (sortBy) {
            case 'newest':
                return new Date(b.created_at) - new Date(a.created_at);
            case 'oldest':
                return new Date(a.created_at) - new Date(b.created_at);
            case 'name_asc':
                return a.name.localeCompare(b.name);
            case 'name_desc':
                return b.name.localeCompare(a.name);
            case 'models_asc':
                return a.models.length - b.models.length;
            case 'models_desc':
                return b.models.length - a.models.length;
            case 'images_asc':
                return getTotalImages(a) - getTotalImages(b);
            case 'images_desc':
                return getTotalImages(b) - getTotalImages(a);
            default:
                return 0;
        }
    });
}

// Display results in grid
function displayResults(projects, totalCount) {
    const resultsGrid = document.getElementById('resultsGrid');
    const resultsCount = document.getElementById('resultsCount');
    const noResults = document.getElementById('noResults');
    const container = document.getElementById('resultsContainer');
    
    resultsCount.textContent = `${totalCount} projects found`;
    
    if (projects.length === 0) {
        container.style.display = 'none';
        noResults.style.display = 'block';
        return;
    }
    
    noResults.style.display = 'none';
    container.style.display = 'block';
    
    let resultsHTML = '';
    projects.forEach(project => {
        const totalImages = getTotalImages(project);
        const createdDate = new Date(project.created_at);
        const isRecent = (Date.now() - createdDate.getTime()) < 7 * 24 * 60 * 60 * 1000; // 7 days
        
        resultsHTML += `
            <div class="project-card-modern ${currentView}">
                <div class="project-card-header">
                    <div class="project-badge ${isRecent ? 'new' : ''}">
                        ${isRecent ? '<i class="fas fa-star"></i> NEW' : ''}
                    </div>
                    <div class="project-actions">
                        <button class="btn-icon small" onclick="quickView('${project.id}')" title="Quick View">
                            <i class="fas fa-eye"></i>
                        </button>
                        <button class="btn-icon small" onclick="exportProject('${project.id}')" title="Export">
                            <i class="fas fa-download"></i>
                        </button>
                    </div>
                </div>
                
                <div class="project-card-content">
                    <h4 class="project-title">${project.name}</h4>
                    <p class="project-description">${project.description || 'No description provided'}</p>
                    
                    <div class="project-stats-modern">
                        <div class="stat-item-modern">
                            <i class="fas fa-users"></i>
                            <span class="stat-value">${project.models.length}</span>
                            <span class="stat-label">Models</span>
                        </div>
                        <div class="stat-item-modern">
                            <i class="fas fa-images"></i>
                            <span class="stat-value">${totalImages}</span>
                            <span class="stat-label">Images</span>
                        </div>
                        <div class="stat-item-modern">
                            <i class="fas fa-calendar"></i>
                            <span class="stat-value">${createdDate.toLocaleDateString()}</span>
                        </div>
                    </div>
                    
                    ${project.models.length > 0 ? `
                        <div class="project-models-preview">
                            <div class="models-scroll">
                                ${project.models.slice(0, 4).map(model => `
                                    <div class="model-chip">
                                        <span class="model-name">${model.model_name}</span>
                                        ${model.instagram_handle ? `
                                            <span class="model-handle">@${model.instagram_handle}</span>
                                        ` : ''}
                                    </div>
                                `).join('')}
                                ${project.models.length > 4 ? `
                                    <div class="more-models">+${project.models.length - 4} more</div>
                                ` : ''}
                            </div>
                        </div>
                    ` : ''}
                </div>
                
                <div class="project-card-footer">
                    <span class="project-status status-${project.status}">
                        <i class="fas fa-circle"></i>
                        ${project.status.charAt(0).toUpperCase() + project.status.slice(1)}
                    </span>
                    <a href="<?php echo site_url('project/view/'); ?>${project.id}" class="btn btn-primary small">
                        View Details
                        <i class="fas fa-arrow-right"></i>
                    </a>
                </div>
            </div>
        `;
    });
    
    resultsGrid.innerHTML = resultsHTML;
}

// Update active filters display
function updateActiveFilters() {
    const activeFilters = document.getElementById('activeFilters');
    const filterTags = document.getElementById('filterTags');
    const filtersCount = document.getElementById('activeFiltersCount');
    
    const activeFiltersList = [];
    
    if (currentFilters.search) {
        activeFiltersList.push({
            type: 'search',
            label: `Search: "${currentFilters.search}"`,
            value: currentFilters.search
        });
    }
    
    if (currentFilters.status.length > 0) {
        activeFiltersList.push({
            type: 'status',
            label: `Status: ${currentFilters.status.join(', ')}`,
            value: currentFilters.status.join(',')
        });
    }
    
    if (currentFilters.dateFrom || currentFilters.dateTo) {
        const from = currentFilters.dateFrom || 'Any';
        const to = currentFilters.dateTo || 'Now';
        activeFiltersList.push({
            type: 'date',
            label: `Date: ${from} to ${to}`,
            value: `${from}-${to}`
        });
    }
    
    if (currentFilters.modelMin > 0 || currentFilters.modelMax < 50) {
        const label = `Models: ${currentFilters.modelMin} to ${currentFilters.modelMax === 50 ? '50+' : currentFilters.modelMax}`;
        activeFiltersList.push({
            type: 'models',
            label: label,
            value: `${currentFilters.modelMin}-${currentFilters.modelMax}`
        });
    }
    
    if (currentFilters.imageCount !== '0') {
        activeFiltersList.push({
            type: 'images',
            label: `Min Images: ${currentFilters.imageCount}+`,
            value: currentFilters.imageCount
        });
    }
    
    filtersCount.textContent = activeFiltersList.length;
    
    if (activeFiltersList.length > 0) {
        filterTags.innerHTML = activeFiltersList.map(filter => `
            <div class="filter-tag-modern">
                <span class="filter-tag-label">${filter.label}</span>
                <button class="filter-tag-remove" onclick="removeFilter('${filter.type}', '${filter.value}')">
                    <i class="fas fa-times"></i>
                </button>
            </div>
        `).join('');
        activeFilters.style.display = 'block';
    } else {
        activeFilters.style.display = 'none';
    }
}

function removeFilter(type, value) {
    switch (type) {
        case 'search':
            clearSearch();
            break;
        case 'status':
            const statuses = value.split(',');
            statuses.forEach(status => {
                const checkbox = document.querySelector(`input[value="${status}"]`);
                if (checkbox) checkbox.checked = false;
                currentFilters.status = currentFilters.status.filter(s => s !== status);
            });
            break;
        case 'date':
            document.getElementById('dateFrom').value = '';
            document.getElementById('dateTo').value = '';
            currentFilters.dateFrom = '';
            currentFilters.dateTo = '';
            break;
        case 'models':
            document.getElementById('modelCountMin').value = 0;
            document.getElementById('modelCountMax').value = 50;
            updateModelRange();
            break;
        case 'images':
            document.querySelector('input[name="imageCount"][value="0"]').checked = true;
            currentFilters.imageCount = '0';
            break;
    }
    applyFilters();
}

function clearAllFilters() {
    // Reset all filters
    clearSearch();
    
    // Reset status checkboxes
    document.querySelectorAll('input[type="checkbox"]').forEach(cb => cb.checked = false);
    currentFilters.status = [];
    
    // Reset date range
    document.getElementById('dateFrom').value = '';
    document.getElementById('dateTo').value = '';
    currentFilters.dateFrom = '';
    currentFilters.dateTo = '';
    
    // Reset model range
    document.getElementById('modelCountMin').value = 0;
    document.getElementById('modelCountMax').value = 50;
    updateModelRange();
    
    // Reset image count
    document.querySelector('input[name="imageCount"][value="0"]').checked = true;
    currentFilters.imageCount = '0';
    
    // Reset sort and page size
    document.getElementById('sortBy').value = 'newest';
    document.getElementById('pageSize').value = '12';
    
    applyFilters();
}

function updateQuickStats() {
    document.getElementById('totalProjects').textContent = allProjects.length;
    const totalModels = allProjects.reduce((sum, project) => sum + project.models.length, 0);
    const totalImages = allProjects.reduce((sum, project) => sum + getTotalImages(project), 0);
    
    document.getElementById('totalModels').textContent = totalModels;
    document.getElementById('totalImages').textContent = totalImages;
}

// View management
function changeView(view) {
    currentView = view;
    const container = document.getElementById('resultsContainer');
    const buttons = document.querySelectorAll('.view-btn');
    
    // Update active button
    buttons.forEach(btn => {
        btn.classList.toggle('active', btn.dataset.view === view);
    });
    
    // Update container class
    container.className = `results-container ${view}-view`;
    
    // Re-apply filters to refresh view
    applyFilters();
}

// Pagination
function updatePagination(totalPages) {
    const pagination = document.getElementById('pagination');
    
    if (totalPages <= 1) {
        pagination.style.display = 'none';
        return;
    }
    
    pagination.style.display = 'flex';
    
    let paginationHTML = '';
    
    // Previous button
    paginationHTML += `
        <button class="pagination-btn ${currentPage === 1 ? 'disabled' : ''}" 
                onclick="changePage(${currentPage - 1})" ${currentPage === 1 ? 'disabled' : ''}>
            <i class="fas fa-chevron-left"></i>
        </button>
    `;
    
    // Page numbers
    const maxVisible = 5;
    let startPage = Math.max(1, currentPage - Math.floor(maxVisible / 2));
    let endPage = Math.min(totalPages, startPage + maxVisible - 1);
    
    if (endPage - startPage + 1 < maxVisible) {
        startPage = Math.max(1, endPage - maxVisible + 1);
    }
    
    for (let i = startPage; i <= endPage; i++) {
        paginationHTML += `
            <button class="pagination-btn ${i === currentPage ? 'active' : ''}" 
                    onclick="changePage(${i})">
                ${i}
            </button>
        `;
    }
    
    // Next button
    paginationHTML += `
        <button class="pagination-btn ${currentPage === totalPages ? 'disabled' : ''}" 
                onclick="changePage(${currentPage + 1})" ${currentPage === totalPages ? 'disabled' : ''}>
            <i class="fas fa-chevron-right"></i>
        </button>
    `;
    
    pagination.innerHTML = paginationHTML;
}

function changePage(page) {
    currentPage = page;
    applyFilters();
    // Scroll to results
    document.querySelector('.modern-results-section').scrollIntoView({ behavior: 'smooth' });
}

// Loading states
function showLoading() {
    document.getElementById('loadingResults').style.display = 'flex';
    document.getElementById('resultsContainer').style.display = 'none';
    document.getElementById('noResults').style.display = 'none';
    document.getElementById('pagination').style.display = 'none';
    
    // Animate progress bar
    let progress = 0;
    const progressBar = document.getElementById('loadingProgress');
    const interval = setInterval(() => {
        progress += Math.random() * 10;
        if (progress >= 90) {
            progress = 90;
            clearInterval(interval);
        }
        progressBar.style.width = progress + '%';
    }, 200);
}

function hideLoading() {
    document.getElementById('loadingResults').style.display = 'none';
    document.getElementById('loadingProgress').style.width = '100%';
}

// Utility functions
function quickView(projectId) {
    // Implement quick view modal
    showNotification('Quick view for project ' + projectId, 'info');
}

function exportProject(projectId) {
    showNotification('Exporting project ' + projectId, 'success');
}

function exportResults() {
    showNotification('Exporting all search results', 'success');
}

function suggestSimilar() {
    showNotification('Showing similar projects', 'info');
}

function showNotification(message, type) {
    // Simple notification implementation
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <div class="notification-content">
            <i class="fas fa-${type === 'success' ? 'check' : 'info'}"></i>
            <span>${message}</span>
        </div>
    `;
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.remove();
    }, 3000);
}
</script>